#  Copyright (C) 2024
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Mike Orlov <m.orlov@abm-jsc.ru>
import logging
from typing import Any, Iterable, TypeAlias

import yarl
from entity_query import Column, Query, Relation, SubQuery, Order, Expression
from entity_query.remote_entity import RemoteEntity
from http_tools import AbmServiceBearerAuthConnector
from init_helpers import custom_dumps

from abstract_entity_connector_abm.actor import Actor
from abstract_entity_connector_abm.utils.make_parse_func import JsonPath, AttrStructure, make_parse_func


logger = logging.getLogger(__name__)


SelectableStructure: TypeAlias = (
    Column |
    tuple['SelectableStructure', ...] |
    list['SelectableStructure'] |
    dict[str, 'SelectableStructure']
)


class AbstractEntityConnector:
    class Config(AbmServiceBearerAuthConnector.Config):
        pass

    class Context(AbmServiceBearerAuthConnector.Context):
        pass

    def __init__(self, config: Config, context: Context):
        self.config = config
        self.context = context
        self.connector = AbmServiceBearerAuthConnector(config, context)

    async def post(self, path: str, actor: Actor, query: Query) -> Any:
        logger.debug(f'POST {str(yarl.URL(self.config.url) / path.lstrip("/"))}')
        logger.debug(custom_dumps(query))
        # noinspection PyTypeChecker
        result = await self.connector.post(path, payload=query, token=actor.token)
        return result

    @classmethod
    def cast_selectable_structure_to_attr_structure(cls, structure: SelectableStructure) -> AttrStructure | None:
        if isinstance(structure, Column):
            return JsonPath([
                parent.key for parent in reversed(cls._get_column_parents(structure)) if isinstance(parent, Relation)
            ] + [structure.key])
        if isinstance(structure, list):
            return [cls.cast_selectable_structure_to_attr_structure(val) for val in structure]
        if isinstance(structure, tuple):
            return tuple(cls.cast_selectable_structure_to_attr_structure(val) for val in structure)
        if isinstance(structure, dict):
            return {key: cls.cast_selectable_structure_to_attr_structure(val) for key, val in structure.items()}

    async def get_by_attr_structure(
            self, actor: Actor, entity: type[RemoteEntity], structure: SelectableStructure,
            filters: list[Expression] | None = None, order: list[Order] | None = None,
            limit: int | None = None, offset: int | None = None, one: bool = False
    ) -> Any:
        # TODO: remove entity from signature
        query = self.get_query_from_attr_structure(entity, structure, filters, order, limit, offset)
        attr_structure = self.cast_selectable_structure_to_attr_structure(structure)
        parse = make_parse_func(attr_structure)
        answer = await self.get(actor, query)
        if one:
            assert len(answer) == 1, f"Expected exactly one {entity.__name__} with {filters}, got {len(answer)}"
            return parse(answer[0])
        result = [parse(val) for val in answer]
        return result

    @classmethod
    def _walk(cls, target: Any) -> Iterable:
        if isinstance(target, (list | set | tuple)):
            for value in target:
                yield from cls._walk(value)
        elif isinstance(target, dict):
            for key, value in target.items():
                yield from cls._walk(key)
                yield from cls._walk(value)
        else:
            yield target

    @classmethod
    def get_query_from_attr_structure(
            cls, entity: type[RemoteEntity], structure: SelectableStructure,
            filters: list[Expression] | None = None, order: list[Order] | None = None,
            limit: int | None = None, offset: int | None = None
    ) -> Query:
        columns: set[Column] = set(filter(lambda x: isinstance(x, Column), cls._walk(structure)))
        result = Query(entity, filters=filters or [], orders=order or [], limit=limit, offset=offset)
        for col in columns:
            cls.insert_column_to_query_attrs(col, result)
        return result

    @classmethod
    def insert_column_to_query_attrs(cls, column: Column, query: Query) -> None:
        current_query: Query | SubQuery = query
        for parent_relation in filter(lambda x: isinstance(x, Relation), reversed(cls._get_column_parents(column))):
            relation_to_subquery = {
                query.over: query for query in filter(lambda x: isinstance(x, SubQuery), current_query.attrs)
            }
            base_relation: Relation = parent_relation(None)
            if base_relation in relation_to_subquery:
                child_subquery = relation_to_subquery[base_relation]
            else:
                child_subquery = SubQuery(over=base_relation)
                current_query.attrs.append(child_subquery)
            current_query = child_subquery
        current_query.attrs.append(column(None))

    @classmethod
    def _get_column_parents(cls, column: Column) -> list[RemoteEntity | type[RemoteEntity] | Relation]:
        result: list[RemoteEntity | type[RemoteEntity] | Relation] = []
        target: Column | RemoteEntity | Relation = column
        while (parent := target.get_parent()) is not None:
            if isinstance(parent, type):
                break
            target = parent
            result.append(parent)
        return result
