#  Copyright (C) 2023
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Revva Konstantin <k.revva@abm-jsc.ru>


import logging
from dataclasses import dataclass, asdict
from typing import Optional, Union
from datetime import datetime
from http_tools.http_server_connector import HttpServerConnector
from init_helpers.dict_to_dataclass import dict_to_dataclass


logger = logging.getLogger(__name__)


@dataclass
class EntityCondition:
    attribute: str
    value: Union[int, float, bool, str, None, list[Union[int, float, bool, str, None]]]
    operator: str = "="


@dataclass
class EntityOrder:
    column: str
    ascending: bool = True


@dataclass
class ReferenceImage:
    id: int
    rtsp_url: str
    file_id: str
    created_at: Optional[int]
    occurred_at: Optional[int]
    camera_id: Optional[int]


class ReferenceImageConnector:
    @dataclass
    class Context(HttpServerConnector.Context):
        project_name: str

    Config = HttpServerConnector.Config

    def __init__(self, config: Config, context: Context):
        self.config = config
        self.context = context
        self._connector = HttpServerConnector(config, context)

    async def get_reference_image_entities(self,
                                           filter_by: Optional[list[EntityCondition]] = None,
                                           search_by: Optional[list[EntityCondition]] = None,
                                           order_by: Optional[list[EntityOrder]] = None,
                                           limit: Optional[int] = None,
                                           offset: Optional[int] = None) -> list[ReferenceImage]:
        payload = {}
        if filter_by is not None:
            payload["filter_by"] = [asdict(filter_by_) for filter_by_ in filter_by]
        if search_by is not None:
            payload["search_by"] = [asdict(search_by_) for search_by_ in search_by]
        if order_by is not None:
            payload["order_by"] = [asdict(order_by_) for order_by_ in order_by]
        if limit is not None:
            payload["limit"] = limit
        if offset is not None:
            payload["offset"] = offset

        answer = await self._connector.post_json("/entity/reference_image/get",
                                                 payload=payload,
                                                 headers=self._construct_headers())
        return [dict_to_dataclass(reference_image, ReferenceImage) for reference_image in answer["result"]]

    async def get_reference_image_file_ids(self, stream_url: str, date: datetime) -> list[str]:
        payload = {"stream_url": stream_url, "iso_date": date.isoformat()}

        answer = await self._connector.get("/reference_image/get",
                                           url_query=payload,
                                           headers=self._construct_headers())
        return answer["result"]

    async def add_reference_image(self,
                                  stream_url: str,
                                  file_id: str,
                                  init_weight: Optional[int] = None,
                                  weight_hour_shift: Optional[int] = None,
                                  weight_month_shift: Optional[int] = None) -> bool:
        payload = {"stream_url": stream_url, "file_id": file_id}
        if init_weight is not None:
            payload["init_weight"] = init_weight
        if weight_hour_shift is not None:
            payload["weight_hour_shift"] = weight_hour_shift
        if weight_month_shift is not None:
            payload["weight_month_shift"] = weight_month_shift

        answer = await self._connector.post_json("/reference_image/add",
                                                 payload=payload,
                                                 headers=self._construct_headers())
        return answer["result"]

    def _construct_headers(self) -> dict[str, str]:
        return {"server_name": self.context.project_name}
