import inspect
import string
from inspect import Signature
from typing import Callable, Tuple

from http_tools import Answer, WrappedAnswer, HttpStatusCode
from http_tools.answer import ExceptionAnswer, JsonableAnswer, ErrorAnswer

from .open_api_server import RpcEndpoint
from .security import Security, get_securities_arg_names, SecurityScheme
from .parameter import SpecParameter, QueryParameter, PathParameter, JsonBodyParameter, CallParameter, RawBodyParameter


def gen_query_rpc_endpoint(
        func: Callable, method: str, path: str, securities: list[Security | SecurityScheme],
        exception_type_to_answer_type: dict[type[Exception], type[ExceptionAnswer]] | None = None,
        unhandled_exception__answer_type: type[Answer] = ExceptionAnswer[HttpStatusCode.InternalServerError],
        arg_name_to_param: dict[str, CallParameter | type[RawBodyParameter] | None] | None = None,
        base_answer_type: type[JsonableAnswer] = WrappedAnswer[HttpStatusCode.OK],
) -> RpcEndpoint:
    """
    Generate RPC endpoint based on function signature:
    - Return type will be used to generated answer type.
    - All missing arguments will be treated as QueryParameters
    """
    return _gen_rpc_endpoint(
        param_type=QueryParameter, func=func, method=method, path=path, securities=securities,
        exception_type_to_answer_type=exception_type_to_answer_type,
        unhandled_exception__answer_type=unhandled_exception__answer_type,
        arg_name_to_param=arg_name_to_param, base_answer_type=base_answer_type
    )


def gen_json_rpc_endpoint(
        func: Callable, method: str, path: str, securities: list[Security | SecurityScheme],
        exception_type_to_answer_type: dict[type[Exception], type[ExceptionAnswer]] | None = None,
        unhandled_exception__answer_type: type[Answer] = ExceptionAnswer[HttpStatusCode.InternalServerError],
        arg_name_to_param: dict[str, CallParameter | type[RawBodyParameter] | None] | None = None,
        base_answer_type: type[JsonableAnswer] = WrappedAnswer[HttpStatusCode.OK],
) -> RpcEndpoint:
    """
    Generate RPC endpoint based on function signature:
    - Return type will be used to generated answer type.
    - All missing arguments will be treated as JsonBodyParameter
    """
    return _gen_rpc_endpoint(
        param_type=JsonBodyParameter, func=func, method=method, path=path, securities=securities,
        exception_type_to_answer_type=exception_type_to_answer_type,
        unhandled_exception__answer_type=unhandled_exception__answer_type,
        arg_name_to_param=arg_name_to_param, base_answer_type=base_answer_type
    )


def _get_path_param_names(path: str) -> set[str]:
    return {i[1] for i in string.Formatter().parse(path) if i[1] is not None}


def _gen_rpc_endpoint(
        param_type: type[SpecParameter],
        func: Callable, method: str, path: str, securities: list[Security | SecurityScheme],
        exception_type_to_answer_type: dict[type[Exception], type[ExceptionAnswer]] | None = None,
        unhandled_exception__answer_type: type[Answer] = ExceptionAnswer[HttpStatusCode.InternalServerError],
        arg_name_to_param: dict[str, CallParameter | type[RawBodyParameter]] | None = None,
        base_answer_type: type[JsonableAnswer] = WrappedAnswer[HttpStatusCode.OK],
) -> RpcEndpoint:
    path_param_names = _get_path_param_names(path)
    security_param_names = get_securities_arg_names(securities)
    signature = Signature.from_callable(func)
    arg_name_to_param = arg_name_to_param or {}
    result_arg_name_to_param = {}
    for arg in signature.parameters.values():
        if arg.annotation == Signature.empty:
            raise TypeError(f"undefined type for arg: {arg}")
        if arg.name in arg_name_to_param:
            if arg_name_to_param[arg.name] is not None:
                result_arg_name_to_param[arg.name] = arg_name_to_param[arg.name]
            continue

        schema = arg.annotation
        if arg.kind == inspect.Parameter.VAR_POSITIONAL:
            schema = list[schema]
        elif arg.kind == inspect.Parameter.VAR_KEYWORD:
            schema = dict[str, schema]

        parameter_kwargs = {"name": arg.name, "schema": schema}
        if arg.default != Signature.empty:
            parameter_kwargs["default"] = arg.default
        if arg.name in path_param_names:
            param = PathParameter(**parameter_kwargs)
        elif arg.name in security_param_names:
            continue
        else:
            # noinspection PyArgumentList
            param = param_type(**parameter_kwargs)
        result_arg_name_to_param[arg.name] = param

    if signature.return_annotation == Signature.empty:
        raise TypeError("undefined return type")

    return_type = signature.return_annotation
    return_type = type[None] if return_type is None else return_type

    return RpcEndpoint(
        func=func, method=method, path=path, securities=securities, arg_name_to_param=result_arg_name_to_param,
        answer_type=base_answer_type[return_type],
        exception_type_to_answer_type=exception_type_to_answer_type,
        unhandled_exception__answer_type=unhandled_exception__answer_type,
    )
