#  Copyright (C) 2024
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Mike Orlov <m.orlov@abm-jsc.ru>
import logging
from dataclasses import dataclass
from typing import Type

from aiohttp import ClientResponseError
from async_tools import AsyncInitable
from dict_caster.extras import first
from file_storage.abstract_file_storage import AbstractFileStorage

from .database import Database, WrappedSession
from .entities.link_info import LinkInfo
from .entities.thumbnail_info import ThumbnailInfo
from .extra import silent, AnySession
from .file_downloader import FileDownloader
from .file_keeper import FileKeeper
from .link_keeper import LinkKeeper, LI
from .memory_zipper import MemoryZipper
from .thumbnail.thumbnail_provider import ThumbnailProvider


logger = logging.getLogger(__name__)


class Controller(AsyncInitable):
    @dataclass
    class Config:
        pass

    @dataclass
    class Context:
        database: Database
        file_storage: AbstractFileStorage
        thumbnail_provider: ThumbnailProvider
        file_downloader: FileDownloader

    def __init__(self,
                 config: Config, context: Context,
                 link_class: Type[LI] = LinkInfo, thumbnail_class: Type[LI] = ThumbnailInfo,
                 ) -> None:
        self.config = config
        self.context = context

        self.file_keeper = FileKeeper(FileKeeper.Context(self.context.database, self.context.file_storage))
        self.link_keeper = LinkKeeper(
            LinkKeeper.Context(self.file_keeper, self.context.database), link_class=link_class
        )
        self.thumbnail_keeper = LinkKeeper(
            LinkKeeper.Context(self.file_keeper, self.context.database), link_class=thumbnail_class
        )
        self.zipper = MemoryZipper(MemoryZipper.Context(self.link_keeper, self.context.database))
        super().__init__()

    async def _async_init(self) -> None:
        await self.file_keeper._set_file_storage_capacity()

    def _publish(self):
        ...

    async def _find_thumbnail_key_by_file_key(self, file_info_key: str, session:  WrappedSession) -> str | None:
        if link_infos_with_same_key := await self.link_keeper.get_link_infos_by_file_info_key(file_info_key, session):
            links_ids = [link.id for link in link_infos_with_same_key]
            if thumbnail_infos := await self.thumbnail_keeper.get_link_infos(links_ids, session, limit=1):
                return thumbnail_infos[0].file_info_key

    async def upload_from(self, file_url: str, session: AnySession | None = None,
                          file_kwargs: dict | None = None) -> LI:
        try:
            file_metadata = await self.context.file_downloader.head(file_url)
        except ClientResponseError as er:
            logger.error(f"Received file_metadata failed: {repr(er)}")
            file_metadata = None
        file_info_key = file_metadata.cleared_etag if file_metadata else None
        if file_info_key and await self.file_keeper.head(file_info_key, nullable=True):
            return await self.add_link(
                file_info_key, file_metadata.filename, extension=file_metadata.extension, session=session
            )
        logger.info(f"file '{file_url}' will be downloaded and uploaded")
        filename, content = await self.context.file_downloader.get(file_url)
        return first(await self.upload(content, filename, session=session, file_kwargs=file_kwargs))

    async def upload(
            self, content: bytes, filename: str, new_link_id: str | None = None, session: AnySession = None,
            is_thumbnail_required: bool = False,
            file_kwargs: dict | None = None) -> tuple[LinkInfo, ThumbnailInfo | None]:
        async with self.context.database.ensure_session(session) as session:
            link = await self.link_keeper.upload(content, filename, link_id=new_link_id, session=session,
                                                 file_kwargs=file_kwargs)
            thumbnail = await self.add_thumbnail(link, content, session) if is_thumbnail_required else None
        return link, thumbnail

    async def add_thumbnail(self, link: LI, content: bytes, session: AnySession | None = None) -> ThumbnailInfo | None:
        async with self.context.database.ensure_session(session) as session:
            if thumbnail_key := await self._find_thumbnail_key_by_file_key(link.file_info_key, session):
                logger.info("Found existing thumbnail, using it")
                return await self.thumbnail_keeper.add(
                    thumbnail_key, link.filename, link.id, link.extension, session
                )
            elif thumbnail := await self.context.thumbnail_provider.provide(link.extension, content):
                logger.info("Existing thumbnail not found, created new")
                return await self.thumbnail_keeper.upload(thumbnail, link.filename, link.id, session)

    async def add_link(
            self, file_info_key: str, filename: str, link_id: str | None = None,
            extension: str | None = None, session: AnySession | None = None, link_kwargs: dict | None = None
    ) -> LI:
        return await self.link_keeper.add(file_info_key, filename, link_id, extension, session, link_kwargs)

    async def get(self, link_id: str, byte_range: range | None = None,
                  session: AnySession = None) -> tuple[LinkInfo, bytes]:
        link_info, file_content = await self.link_keeper.get(link_id, byte_range=byte_range, session=session)
        return link_info, file_content

    async def get_thumbnail(self, link_id: str, byte_range: range | None = None,
                            session: AnySession = None) -> tuple[LinkInfo, bytes]:
        link_info, file_content = await self.thumbnail_keeper.get(link_id, byte_range=byte_range, session=session)
        return link_info, file_content

    async def read(self, file_info_key: str, byte_range: range | None = None) -> bytes:
        return await self.link_keeper.read(file_info_key, byte_range)

    async def head(self, link_id: str, session: AnySession = None) -> LinkInfo:
        return await self.link_keeper.head(link_id, session=session)

    async def delete(self, link_id: str, session: AnySession = None) -> LinkInfo:
        async with self.context.database.ensure_session(session) as session:
            result = await self.link_keeper.delete(link_id, session)
            await silent(self.thumbnail_keeper.delete(link_id, session))
        return result

    async def zip_files(self, link_ids: list[str], session: AnySession = None) -> tuple[LinkInfo, bytes]:
        return await self.zipper.zip_files(link_ids, session=session)
