import inspect
import string
from inspect import Signature
from typing import Callable

from http_tools import Answer, WrappedAnswer, HttpStatusCode
from http_tools.answer import ExceptionAnswer

from .open_api_server import RpcEndpoint
from .security import Security, get_securities_arg_names, SecurityScheme
from .parameter import SpecParameter, QueryParameter, PathParameter, JsonBodyParameter


def gen_query_rpc_endpoint(
        func: Callable, method: str, path: str, securities: list[Security | SecurityScheme],
        exception_type_to_answer_type: dict[type[Exception], type[ExceptionAnswer]],
        unhandled_exception__answer_type: type[Answer] = ExceptionAnswer[HttpStatusCode.InternalServerError],
        skip_param_names: list[str] | None = None,
) -> RpcEndpoint:
    return _gen_rpc_endpoint(
        param_type=QueryParameter, func=func, method=method, path=path, securities=securities,
        exception_type_to_answer_type=exception_type_to_answer_type,
        unhandled_exception__answer_type=unhandled_exception__answer_type,
        skip_param_names=skip_param_names
    )


def gen_json_rpc_endpoint(
        func: Callable, method: str, path: str, securities: list[Security | SecurityScheme],
        exception_type_to_answer_type: dict[type[Exception], type[ExceptionAnswer]],
        unhandled_exception__answer_type: type[Answer] = ExceptionAnswer[HttpStatusCode.InternalServerError],
        skip_param_names: list[str] | None = None,
) -> RpcEndpoint:
    return _gen_rpc_endpoint(
        param_type=JsonBodyParameter, func=func, method=method, path=path, securities=securities,
        exception_type_to_answer_type=exception_type_to_answer_type,
        unhandled_exception__answer_type=unhandled_exception__answer_type,
        skip_param_names=skip_param_names
    )


def _get_path_param_names(path: str) -> set[str]:
    return {i[1] for i in string.Formatter().parse(path) if i[1] is not None}


def _gen_rpc_endpoint(
        param_type: type[SpecParameter],
        func: Callable, method: str, path: str, securities: list[Security | SecurityScheme],
        exception_type_to_answer_type: dict[type[Exception], type[ExceptionAnswer]],
        unhandled_exception__answer_type: type[Answer] = ExceptionAnswer[HttpStatusCode.InternalServerError],
        skip_param_names: list[str] | None = None,
) -> RpcEndpoint:
    skip_param_names = skip_param_names or []
    path_param_names = _get_path_param_names(path)
    security_param_names = get_securities_arg_names(securities)
    signature = Signature.from_callable(func)
    arg_name_to_param = {}
    for param in signature.parameters.values():
        if param.annotation == Signature.empty:
            raise TypeError(f"undefined type for arg: {param}")
        if param.name in skip_param_names:
            continue

        schema = param.annotation
        if param.kind == inspect.Parameter.VAR_POSITIONAL:
            schema = list[schema]
        elif param.kind == inspect.Parameter.VAR_KEYWORD:
            schema = dict[str, schema]

        parameter_kwargs = {"name": param.name, "schema": schema}
        if param.default != Signature.empty:
            parameter_kwargs["default"] = param.default
        if param.name in path_param_names:
            param = PathParameter(**parameter_kwargs)
        elif param.name in security_param_names:
            continue
        else:
            # noinspection PyArgumentList
            param = param_type(**parameter_kwargs)
        arg_name_to_param[param.name] = param

    if signature.return_annotation == Signature.empty:
        raise TypeError("undefined return type")

    return_type = signature.return_annotation
    return_type = type[None] if return_type is None else return_type

    return RpcEndpoint(
        func=func, method=method, path=path, securities=securities, arg_name_to_param=arg_name_to_param,
        answer_type=WrappedAnswer[HttpStatusCode.OK][return_type],
        exception_type_to_answer_type=exception_type_to_answer_type,
        unhandled_exception__answer_type=unhandled_exception__answer_type,
    )
