#  Copyright (C) 2021
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors:
#  Vasya Svintsov <v.svintsov@techokert.ru>,
#  Alexander Medvedev <a.medvedev@abm-jsc.ru>,
#  Andrey Vaydich <a.vaydich@abm-jsc.ru>

import logging
import typing
from dataclasses import dataclass

from async_tools import AsyncInitable
from file_storage.abstract_file_storage import AbstractFileStorage
from file_storage.exceptions import FileAlreadyExists, PathDoesNotExists, FileStorageFull
from sqlalchemy import select, func
from sqlalchemy.exc import SQLAlchemyError

from .extra import AnySession
from .entities.file_info import FileInfo
from .database import Database, WrappedSession
from .tools.key_lock import KeyLock

logger = logging.getLogger(__name__)
FI = typing.TypeVar('FI', bound=FileInfo)


class FileKeeper(typing.Generic[FI], AsyncInitable):
    @dataclass
    class Config:
        pass

    @dataclass
    class Context:
        database: Database
        file_storage: AbstractFileStorage

    def __init__(self, context: Context, config: Config = None, file_class: type[FI] = FileInfo) -> None:
        self.config = config or self.Config()
        self.context = context
        self.file_class = file_class
        self._file_lock = KeyLock()
        self.storage_total_bytes: int | None = None
        super().__init__()

    async def _async_init(self) -> None:
        self.storage_total_bytes = await self.get_total_bytes() or 0

    async def _save_to_file_storage(self, content: bytes, key: str) -> None:
        try:
            await self.context.file_storage.save(key, content)
        except FileAlreadyExists as e:
            if await self._load_from_file_storage(key) == content:
                logger.warning("File already exists, but matches current. Possible index lost")
            else:
                self.storage_total_bytes -= len(content)
                raise

    async def _load_from_file_storage(self, key: str, byte_range: range | None = None) -> bytes:
        byte_range = range(0, -1) if byte_range is None else byte_range
        return await self.context.file_storage.load(key, byte_range.start, byte_range.stop)

    async def _delete_from_file_storage(self, key: str) -> None:
        try:
            await self.context.file_storage.delete(key)
        except PathDoesNotExists as e:
            logger.warning("File to delete does not exist in storage. Possible index lost")

    async def _check_storage_limit(self, content: bytes) -> None:
        content_size = len(content)
        size_max_b = self.context.file_storage.config.size_max_b
        free_space = max(size_max_b - self.storage_total_bytes, 0) # TODO: attr size_b will be added in library 'file-storage-abm'
        logger.info(f"Free storage size {free_space} bytes")
        if content_size > free_space:
            raise FileStorageFull(f'Not enough disk space. Free storage size {free_space} bytes, '
                                  f'but need {content_size} bytes')
        self.storage_total_bytes += content_size

    async def _get_bytes_info(self, key: str, session: WrappedSession, nullable: bool = False) -> FI | None:
        # noinspection PyTypeChecker
        query = select(self.file_class).where(self.file_class.key == key)
        return await (session.scalar_or_none(query) if nullable else session.scalar(query))

    async def _get_bytes_infos(self, keys: list[str], session: WrappedSession) -> list[FI]:
        # noinspection PyUnresolvedReferences
        return await session.scalars(select(self.file_class).where(self.file_class.key.in_(keys)))

    async def get_total_bytes(self, session: AnySession = None) -> int | None:
        # noinspection PyTypeChecker
        async with self.context.database.ensure_session(session) as session:
            return await session.scalar_or_none(select(func.sum(self.file_class.size)))

    async def upload(self, content: bytes, session: AnySession = None, file_kwargs: dict | None = None) -> FI:
        logger.info(f'Uploading file, size: {len(content)} bytes')
        new_info: FI = self.file_class.from_bytes(content, **(file_kwargs or {}))
        new_key = new_info.key
        async with self._file_lock.restrict(new_key), self.context.database.ensure_session(session) as session:
            session: WrappedSession
            if present_info := await self._get_bytes_info(new_key, session, nullable=True):
                logger.info(f'Attempt to upload file identical to present with key: {new_key}')
                assert present_info == new_info, f"{new_info=} has same key as {present_info=}, but differs"
            else:
                logger.info(f'Not found saved file with key {new_key}. It will be uploaded')
                await self._check_storage_limit(content)
                await self._save_to_file_storage(content, new_key)
                try:
                    await session.add_and_flush(new_info)
                except SQLAlchemyError:
                    self.storage_total_bytes -= len(content)
        return new_info

    async def get(self, key: str, byte_range: range | None = None, session: AnySession = None) -> tuple[FI, bytes]:
        logger.debug(f'Getting file with {key=}')
        async with self._file_lock.restrict(key), self.context.database.ensure_session(session) as session:
            present_bytes: FI = await self._get_bytes_info(key, session)
        return present_bytes, await self.read(present_bytes.key, byte_range)

    async def read(self, key: str, byte_range: range | None = None) -> bytes:
        logger.debug(f'Reading file with {key=}')
        content = await self._load_from_file_storage(key, byte_range)
        return content

    async def head(self, key: str, session: AnySession = None, nullable: bool = False) -> FI | None:
        logger.info(f'Getting info with {key=}')
        async with self.context.database.ensure_session(session) as session:
            return await self._get_bytes_info(key, session, nullable)

    async def list(self, keys: list[str], session: AnySession = None) -> list[str]:
        logger.info(f'Listing file with specific {len(keys)} keys ')
        async with self.context.database.ensure_session(session) as session:
            infos: list[FI] = await self._get_bytes_infos(keys, session)
        return [info.key for info in infos]

    async def delete(self, key: str, session: AnySession = None) -> FI:
        logger.info(f'Deleting file with key: {key=}')
        async with self._file_lock.restrict(key), self.context.database.ensure_session(session) as session:
            session: WrappedSession
            info: FI = await self.head(key, session=session)
            await self._delete_from_file_storage(info.key)
            await session.delete_and_flush(info)
            self.storage_total_bytes -= info.size
        return info

    def key(self, content: bytes) -> str:
        return self.file_class.from_bytes(content).key
