#  Copyright (C) 2021
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Vasya Svintsov <v.svintsov@techokert.ru>

import asyncio
import logging
import os
from concurrent.futures import ThreadPoolExecutor
from dataclasses import dataclass
from pathlib import Path
from typing import Optional
import aiofiles.os
from aiofile import AIOFile, Writer
from ..utils.to_list import to_list

from ..abstract_file_storage import AbstractFileStorage, FileSource, FileDeleteResult, FileDeleteStatus, \
    EmptyFileSourcesList
from ..exceptions import FileAlreadyExists, PathDoesNotExists, InsecurePath

logger = logging.getLogger(__name__)


class DiscFileStorage(AbstractFileStorage):
    @dataclass
    class Config:
        location: str
        max_deleting_workers: Optional[int] = None

    def __init__(self, config: Config):
        logger.info(f"{type(self).__name__} init. Location: {config.location}")
        self._config = config

    async def save(self, content: bytes, relative_path: str, name: str, allow_rewrite: bool = False) -> None:
        dir_path = f"{self._config.location}/{relative_path}"
        if not os.path.exists(dir_path):
            logger.info(f"Path = {dir_path} does not exists. Will be created")
            os.makedirs(dir_path)

        full_path = f"{dir_path}/{name}"
        logger.info(f"Full path = {full_path}")
        if Path(full_path).exists() and not allow_rewrite:
            raise FileAlreadyExists(full_path)

        logger.info(f"Saving file along the path: {full_path}")
        async with AIOFile(full_path, 'wb') as file:
            writer = Writer(file)
            await writer(content)

    async def load(self, relative_path: str, name: str, offset: int = 0, size: int = -1) -> bytes:
        dir_path = f"{self._config.location}/{relative_path}"
        full_path = f"{dir_path}/{name}"
        if not Path(full_path).exists():
            raise PathDoesNotExists(full_path)

        if not os.path.realpath(full_path).startswith(os.path.realpath(dir_path)):
            raise InsecurePath(full_path)

        logger.info(f"Loading file along the path: {full_path}")
        async with AIOFile(full_path, 'rb') as file:
            content = await file.read(size, offset)

        return content

    async def delete(self, file_sources: list[FileSource] | FileSource) -> list[FileDeleteResult]:
        file_sources = to_list(file_sources)
        if not file_sources:
            return EmptyFileSourcesList

        dir_paths = set()
        file_full_paths = []
        for file_source in file_sources:
            dir_path = f"{self._config.location}/{file_source.relative_path}"
            full_path = f"{dir_path}/{file_source.name}"
            file_full_paths.append(full_path)

        executor = ThreadPoolExecutor(self._config.max_deleting_workers)
        delete_file_tasks = [self._delete_file(full_path, executor) for full_path in file_full_paths]
        delete_file_task_results = await asyncio.gather(*delete_file_tasks, return_exceptions=True)
        file_delete_results = []
        for file_source, delete_file_task_result in zip(file_sources, delete_file_task_results):
            if delete_file_task_result is None:
                dir_paths.add(f"{self._config.location}/{file_source.relative_path}")
                status = FileDeleteStatus.OK
            else:
                status = FileDeleteStatus.FAILED
            file_delete_results.append(
                FileDeleteResult(file_source=file_source, status=status, reason=repr(delete_file_task_result))
            )
        logger.debug(f"File delete results: {file_delete_results}")

        await self._delete_empty_directories(dir_paths)
        return file_delete_results

    @staticmethod
    async def _delete_file(full_path: str, executor: ThreadPoolExecutor) -> None:
        if not Path(full_path).exists():
            raise PathDoesNotExists(full_path)
        await aiofiles.os.remove(full_path, executor=executor)

    @staticmethod
    async def _delete_empty_directories(dir_paths: set[str]) -> None:
        empty_dir_paths = [dir_path for dir_path in dir_paths if not await aiofiles.os.listdir(dir_path)]
        if not empty_dir_paths:
            return

        delete_dir_tasks = [aiofiles.os.rmdir(dir_path) for dir_path in empty_dir_paths]
        delete_dir_task_results = await asyncio.gather(*delete_dir_tasks, return_exceptions=True)

        dir_path_to_exception = {}
        successfully_deleted_dir_paths = []
        for dir_path, delete_dir_task_result in zip(empty_dir_paths, delete_dir_task_results):
            if delete_dir_task_result is None:
                successfully_deleted_dir_paths.append(dir_path)
            else:
                dir_path_to_exception[dir_path] = delete_dir_task_result
        if successfully_deleted_dir_paths:
            logger.debug(f"Successfully deleted empty directories: {successfully_deleted_dir_paths}")
        if dir_path_to_exception:
            logger.warning(f"Failed deleting empty directories: {dir_path_to_exception}")

    async def check_file_existence(self, relative_path: str, name: str) -> bool:
        full_path = f"{self._config.location}/{relative_path}/{name}"
        return Path(full_path).exists()
