#  Copyright (C) 2024
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Mike Orlov <m.orlov@abm-jsc.ru>
from urllib.parse import urlparse, parse_qs

from prometheus_tools.metrics_registry import MetricsRegistry

from .abstract_file_storage import AbstractFileStorage
from .disc_file_storage.disc_file_storage import DiscFileStorage
from .s3_file_storage.s3_file_storage import S3FileStorage


def _get_disc_file_storage(
        location: str, max_workers: str | None = None, metrics: MetricsRegistry | None = None
) -> DiscFileStorage:
    config = DiscFileStorage.Config(location, max_workers=max_workers if max_workers is None else int(max_workers))
    context = DiscFileStorage.Context(metrics=metrics or MetricsRegistry(MetricsRegistry.Config()))
    return DiscFileStorage(config, context)


def _get_s3_file_storage(access_key: str, secret_key: str, address: str | None = None, region_name: str | None = None,
                         bucket_name: str | None = None, root_dir: str | None = None,
                         metrics: MetricsRegistry | None = None) -> S3FileStorage:
    kwargs = {"access_key": access_key, "secret_key": secret_key}
    kwargs |= {"address": address} if address is not None else {}
    kwargs |= {"region_name": region_name} if region_name is not None else {}
    kwargs |= {"bucket_name": bucket_name} if bucket_name is not None else {}
    kwargs |= {"root_dir": root_dir} if root_dir is not None else {}
    config = S3FileStorage.Config(**kwargs)
    context = S3FileStorage.Context(metrics=metrics or MetricsRegistry(MetricsRegistry.Config()))
    return S3FileStorage(config, context)


def _pop_get_first(query_params: dict[str, list[str]], key: str) -> str | None:
    values = query_params.pop(key, [])
    if values:
        assert len(values) == 1, f'Got multiple values of {key!r}: {len(values)}, expected 1'
        return values[0]


FILE_URL_FORMAT = 'file://<path>[?max_workers=<max_workers>]'
FILE_URL_EXAMPLE = 'file:///tmp/storage'
S3_URL_FORMAT = '(s3|s3s)://<access_key>:<secret_key>@[<address>][:<port>]/[<region>]?[bucket=<bucket>]&[root_dir=<root_dir>]'
S3_URL_EXAMPLE = 's3://access:secret@localhost:7480/Default?bucket=abm-bucket&root_dir=abm-dir'


def get_file_storage(url: str, metrics: MetricsRegistry | None = None) -> AbstractFileStorage:
    parsed_url = urlparse(url)
    query_params: dict[str, list[str]] = parse_qs(parsed_url.query)
    if parsed_url.scheme == "file":
        assert not parsed_url.netloc, (f"File url should not have user, password, host or port, "
                                       f"got: {parsed_url.netloc}, format: {S3_URL_FORMAT}, example: {S3_URL_EXAMPLE}")
        result = _get_disc_file_storage(
            location=parsed_url.path, max_workers=_pop_get_first(query_params, 'max_workers'), metrics=metrics
        )
    elif parsed_url.scheme == "s3" or parsed_url.scheme == "s3s":
        assert parsed_url.username, f'Missing access_key, format: {S3_URL_FORMAT}, example: {S3_URL_EXAMPLE}'
        assert parsed_url.password, f'Missing secret_key, format: {S3_URL_FORMAT}, example: {S3_URL_EXAMPLE}'
        is_secure = parsed_url.scheme == "s3s"
        address = parsed_url.hostname or ''
        address += f':{parsed_url.port}' if parsed_url.port else ''
        address = f'http{"s" if is_secure else ""}://{address}' if address else None
        path = parsed_url.path.lstrip("/")
        result = _get_s3_file_storage(
            access_key=parsed_url.username, secret_key=parsed_url.password, address=address, region_name=path or None,
            bucket_name=_pop_get_first(query_params, 'bucket'), root_dir=_pop_get_first(query_params, 'root_dir'),
            metrics=metrics
        )
    else:
        raise TypeError(f"Attempt to get storage from unsupported scheme '{parsed_url.scheme}', supported: file|s3|s3s")
    assert not query_params, f"Got unexpected query parameters: {set(query_params)}"
    return result
