import re
from typing import ClassVar, Any

from .query_token import QueryToken


CAMEL_PAT = re.compile(r'(?<!^)(?=[A-Z])')


def camel_to_snake(name: str) -> str:
    return CAMEL_PAT.sub('_', name).lower()


def bind_columns_and_relations_to_self(self, value):
    from .relation import Relation
    from .column import Column
    if isinstance(value, (Column | Relation)):
        return value(parent=self)
    return value


class RemoteEntityType(type):
    def __getattribute__(self, item: str) -> Any:
        result = super().__getattribute__(item)
        if not item.startswith("_") and isinstance(result, QueryToken):
            result = bind_columns_and_relations_to_self(self, result)
        return result


class RemoteEntity(metaclass=RemoteEntityType):
    def __init__(self, parent: 'Relation | None' = None):
        self._parent = parent

    def get_parent(self) -> 'Relation | None':
        return self._parent

    def __init_subclass__(cls, remote_name: str | None = None) -> None:
        super().__init_subclass__()
        if remote_name:
            cls._remote_name = remote_name
        if not getattr(cls, '_remote_name', None):
            cls._remote_name = camel_to_snake(cls.__name__)

    _remote_name: ClassVar[str]

    def __getattribute__(self, item: str) -> Any:
        result = super().__getattribute__(item)
        if not item.startswith("_") and isinstance(result, QueryToken):
            result = bind_columns_and_relations_to_self(self, result)
        return result

    @classmethod
    def get_remote_name(cls) -> str:
        return cls._remote_name

    def __str__(self) -> str:
        cls = type(self)
        return f'{cls.__name__}(parent={self._parent})'

    def subquery(
            self,
            attrs: 'list[Expression] | Selectable | None' = None,
            vars: dict[str, 'Expression'] | None = None,
            filters: list['Expression'] | None = None,
            searches: list['Expression'] | None = None,
            orders: list['Order'] | None = None,
            limit: int | None = None,
            offset: int | None = None,
    ) -> 'SubQuery':
        assert self._parent is not None
        from .subquery import SubQuery
        return SubQuery(
            over=self._parent, attrs=attrs or [], vars=vars or {}, filters=filters or [],
            searches=searches or [], orders=orders or [], limit=limit, offset=offset
        )

    __repr__ = __str__
