import asyncio
import datetime
from logging import getLogger
from typing import Callable

from async_tools import acall


logger = getLogger(__name__)


class Periodic:
    def __init__(self, job: Callable, period: datetime.timedelta, is_active: bool = True):
        self.job = job
        self.period = period
        self.is_active = is_active
        self.future = None
        if is_active:
            self.start()

    def start(self):
        if self.future is not None:
            logger.warning("start called on already running Periodic, ignored")
            return
        logger.info("periodic started")
        self.future = asyncio.ensure_future(self.loop())

    async def loop(self):
        await asyncio.sleep(self.period.total_seconds())
        while self.is_active:
            logger.info(f"starting periodic job: {self.job}")
            try:
                await acall(self.job)
            except Exception as e:
                logger.error(f"periodic job raised an error: {repr(e)}")
                logger.exception(e)
            logger.debug(f"finished periodic job: {self.job}")
            if not self.is_active:
                break
            logger.debug(f"sleep {self.period.total_seconds()} sec till next job call")
            await asyncio.sleep(self.period.total_seconds())
        logger.warning("periodic ended:")
