#  Copyright (C) 2023
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Vasya Svintsov <v.svintsov@techokert.ru>
from dataclasses import dataclass, fields
from typing import TypeVar, Final, Type

from .camel_case_to_underscore import camel_case_to_underscore


ENTITY_NAME_FIELD: Final[str] = '__entity_name__'
Entity = TypeVar('Entity')
RELATIONSHIP_COLUMN_SEPARATOR: Final[str] = '.'


def entity_dataclass(class_: type = None, *, entity_name: str = None, **kwargs):
    def _entity_dataclass(cls):
        entity = dataclass(cls, **kwargs)
        setattr(entity, ENTITY_NAME_FIELD, entity_name if entity_name else camel_case_to_underscore(cls.__name__))
        return entity

    if class_ is None:
        return _entity_dataclass

    return _entity_dataclass(class_)


def prepare_entity_column_names(entity: Type[Entity], prefix: str = '') -> list[str]:
    result = []
    for field in fields(entity):
        type_origin = getattr(field.type, '__origin__', None)
        type_args = getattr(field.type, '__args__', None)
        if isinstance(type_origin, type):
            result += prepare_entity_column_names(type_args[0], field.name + RELATIONSHIP_COLUMN_SEPARATOR)
        else:
            result.append(prefix + field.name)
    return result


def is_entity_dataclass(entity) -> bool:
    cls = entity if isinstance(entity, type) else type(entity)
    return hasattr(cls, ENTITY_NAME_FIELD)
