#  Copyright (C) 2023
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Vasya Svintsov <v.svintsov@techokert.ru>

from dataclasses import dataclass, field
from typing import List

from prometheus_client import CollectorRegistry, REGISTRY, Counter, Gauge, Histogram
from prometheus_client.utils import INF


class OutgoingHttpRequestsMetricsCollector:
    class Label:
        method = "method"
        url = "url"
        status_code = "status_code"

    @dataclass
    class Context:
        registry: CollectorRegistry = REGISTRY

    @dataclass
    class Config:
        latency_buckets: List[float] = field(default_factory=lambda: [.1, .25, .5, .75, 1.0, 2.5, 5, 7.5, 10.0, INF])

    def __init__(self, config: Config, context: Context) -> None:
        self.config = config
        self.context = context

        self.total_requests_counter = Counter(
            "total_outgoing_requests",
            "Total number of HTTP requests sent",
            labelnames=[self.Label.method, self.Label.url, self.Label.status_code],
            registry=self.context.registry,
        )

        self.active_requests_gauge = Gauge(
            "active_outgoing_requests",
            "Number of active HTTP requests",
            labelnames=[self.Label.method, self.Label.url],
            registry=self.context.registry
        )

        self.requests_latency_histogram = Histogram(
            "outgoing_requests_latency",
            "Http request latency",
            labelnames=[self.Label.method, self.Label.url, self.Label.status_code],
            registry=self.context.registry,
            buckets=self.config.latency_buckets
        )
