#  Copyright (C) 2024
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Mike Orlov <m.orlov@abm-jsc.ru>
import abc
from dataclasses import dataclass
from typing import ClassVar

from run_markdown.utils import compare_exceptions


@dataclass(frozen=True, slots=True)
class ComparisonLine(abc.ABC):
    left: str
    right: str
    exception: Exception | None = None
    MAX_DEPTH: ClassVar[int] = 32

    # noinspection PyPropertyDefinition
    @classmethod
    @property
    @abc.abstractmethod
    def separator(cls) -> str:
        raise NotImplementedError

    def to_str(self, depth: int = 1) -> str:
        result = ' ...ok' if self.exception is None else f' ...BAD: {self.exception}'
        # return ' ' * depth + f'{self.left} // {self.right} ...{result}'
        spaces = ' ' * depth
        # extra_spaces = ' ' * (self.MAX_DEPTH - depth)
        left = spaces + self.left
        extra_left_spaces = ' ' * (self.MAX_DEPTH - len(left))
        right = spaces + self.right
        extra_right_spaces = ' ' * (self.MAX_DEPTH - len(right))
        return left + extra_left_spaces + self.separator + right + extra_right_spaces + result

    def __eq__(self, other: 'ComparisonLine') -> bool:
        if type(other) is not type(self) or not compare_exceptions(self.exception, other.exception):
            return False
        return self.left == other.left and self.right == other.right


@dataclass(frozen=True, slots=True)
class JsonComparisonLine(ComparisonLine):
    separator: ClassVar[str] = '//'

    def __eq__(self, other: 'ComparisonLine') -> bool:
        if type(other) is not type(self) or not compare_exceptions(self.exception, other.exception):
            return False
        return self.left == other.left and self.right == other.right


@dataclass(frozen=True, slots=True)
class TextComparisonLine(ComparisonLine):
    separator: ClassVar[str] = '//#--#//'
    MAX_DEPTH: ClassVar[int] = 64

    def __eq__(self, other: 'ComparisonLine') -> bool:
        if type(other) is not type(self) or not compare_exceptions(self.exception, other.exception):
            return False
        return self.left == other.left and self.right == other.right
