#  Copyright (C) 2021-2025
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors:
#  Vasya Svintsov <v.svintsov@techokert.ru>,
#  Alexander Medvedev <a.medvedev@abm-jsc.ru>,
#  Andrey Vaydich <a.vaydich@abm-jsc.ru>,
#  Ruslan Albakov <r.albakov@abm-jsc.ru>
import logging
import re
import traceback
from dataclasses import dataclass

import yarl
from aiohttp.hdrs import RANGE, CONTENT_LENGTH, IF_NONE_MATCH
from aiohttp.web_exceptions import HTTPUnprocessableEntity
from dict_caster import Item, DictCaster
from http_tools import HttpServer, Answer, FileAnswer, WrappedAnswer, HttpStatusCode
from http_tools.request import IncomingRequest
from sqlalchemy_tools.database_connector.database_connector import DatabaseConnector

from .controller import Controller
from .extra import assert_raise
from .tools.error_middleware import error_middleware
from .tools.file_extension import define_file_mime_type_by_extension


logger = logging.getLogger(__name__)


class HttpHandler:
    @dataclass
    class Context:
        http_server: HttpServer
        controller: Controller

    @dataclass(frozen=True, slots=True)
    class _FileId:
        bucket_prefix: str
        id: str

    def __init__(self, context: Context, path_prefix: str = '/') -> None:
        self.context = context
        self.bucket_prefix = context.controller.context.file_storage.config.root_dir
        self._register_handlers(path_prefix)

    def _register_handlers(self, path_prefix: str = '/'):
        url = yarl.URL(path_prefix)
        self.context.http_server.register_handler(str(url / 'file/add_from'), self.add_from)
        self.context.http_server.register_handler(str(url / 'file/add'), self.add)
        self.context.http_server.register_handler(str(url / 'file/get'), self.get)
        self.context.http_server.register_handler(str(url / 'file/get'), self.head, {"HEAD"})
        self.context.http_server.register_handler(str(url / 'file/preview'), self.get_thumbnail)
        self.context.http_server.register_handler(str(url / 'file/zip'), self.zip_files)
        self.context.http_server.register_handler(str(url / 'file/info'), self.info)
        self.context.http_server.register_handler(str(url / 'file/delete'), self.delete)
        self.context.http_server.register_handler(str(url / 'file/copy/from'), self.copy_from)

    @error_middleware
    async def add_from(self, request: IncomingRequest) -> Answer:
        file_url = DictCaster(
            Item('file_url', str),
        ).cast_and_return(request.key_value_arguments)
        link = await self.context.controller.upload_from(file_url)
        link.id = self.get_bucket_prefix_with_file_id(link.id)
        link.file_info_key = self.get_bucket_prefix_with_file_id(link.file_info_key)
        return WrappedAnswer(link)

    @error_middleware
    async def add(self, request: IncomingRequest) -> Answer:
        if file := request.key_value_arguments.get('file'):  # multipart/form-data
            filename = file.filename
            content = file.content
        elif request.payload:  # application/octet-stream
            filename = DictCaster(Item('filename', str)).cast_and_return(request.key_value_arguments)
            content = request.payload
        else:
            logger.debug(request)
            raise HTTPUnprocessableEntity(reason='FileMissing')

        is_thumbnail_required = DictCaster(
            Item('is_thumbnail_required', bool, default=False),
        ).cast_and_return(request.key_value_arguments)

        try:
            link_info, thumbnail_info = await self.context.controller.upload(
                content=content, filename=filename, is_thumbnail_required=is_thumbnail_required,
            )
            link_info.id = self.get_bucket_prefix_with_file_id(link_info.id)
            link_info.file_info_key = self.get_bucket_prefix_with_file_id(link_info.file_info_key)
            thumbnail_info.id = self.get_bucket_prefix_with_file_id(thumbnail_info.id)
            thumbnail_info.file_info_key = self.get_bucket_prefix_with_file_id(thumbnail_info.file_info_key)
        except Exception:
            traceback.print_exc()
            raise
        return WrappedAnswer({"original": link_info, "thumbnail": thumbnail_info})

    @error_middleware
    async def get(self, request: IncomingRequest) -> Answer:
        link_id = DictCaster(Item('id', str)).cast_and_return(request.key_value_arguments)
        link_id = self.get_file_id(link_id).id
        byte_range = _parse_range_header(request.metadata.get_header(RANGE))
        possible_etags = _parse_if_none_match_header(request.metadata.get_header(IF_NONE_MATCH))
        link_info = await self.context.controller.head(link_id)
        assert_raise(link_info, FileNotFoundError, link_id)
        headers = generate_cache_headers(link_info.file_info_key)
        if link_info.file_info_key in possible_etags:
            return Answer(None, HttpStatusCode.NotModified, headers=headers)
        payload = await self.context.controller.read(link_info.file_info_key, byte_range)
        return FileAnswer(payload=payload, file_name=link_info.filename,
                          content_type=define_file_mime_type_by_extension(link_info.extension),
                          headers=headers)

    @error_middleware
    async def head(self, request: IncomingRequest) -> Answer:
        link_id = DictCaster(Item('id', str)).cast_and_return(request.key_value_arguments)
        link_id = self.get_file_id(link_id).id
        link_info = await self.context.controller.head(link_id)
        assert_raise(link_info, FileNotFoundError, link_id)
        return FileAnswer(payload=None, file_name=link_info.filename,
                          content_type=define_file_mime_type_by_extension(link_info.extension),
                          headers={CONTENT_LENGTH: str(link_info.file_info.size),
                                   **generate_cache_headers(link_info.file_info_key)})

    @error_middleware
    async def zip_files(self, request: IncomingRequest) -> Answer:
        ids = DictCaster(Item('ids', list[str])).cast_and_return(request.key_value_arguments)
        ids = [self.get_file_id(id).id for id in ids]
        link_info, payload = await self.context.controller.zip_files(ids)
        return FileAnswer(payload=payload, file_name=link_info.filename,
                          content_type=define_file_mime_type_by_extension(link_info.extension))

    @error_middleware
    async def get_thumbnail(self, request: IncomingRequest) -> Answer:
        link_id = DictCaster(Item('id', str)).cast_and_return(request.key_value_arguments)
        link_info, payload = await self.context.controller.get_thumbnail(link_id)
        return FileAnswer(payload=payload, file_name=link_info.filename,
                          content_type=define_file_mime_type_by_extension(link_info.extension),
                          headers=generate_cache_headers(link_info.file_info_key))

    @error_middleware
    async def info(self, request: IncomingRequest) -> Answer:
        link_id = DictCaster(Item('id', str)).cast_and_return(request.key_value_arguments)
        link_id = self.get_file_id(link_id).id
        link_info = await self.context.controller.head(link_id)
        link_info.id = self.get_bucket_prefix_with_file_id(link_info.id)
        link_info.file_info_key = self.get_bucket_prefix_with_file_id(link_info.file_info_key)
        return WrappedAnswer(link_info)

    @error_middleware
    async def delete(self, request: IncomingRequest) -> Answer:
        link_id = DictCaster(Item('id', str)).cast_and_return(request.key_value_arguments)
        link_info = await self.context.controller.delete(link_id)
        return WrappedAnswer(link_info)

    @error_middleware
    async def copy_from(self, request: IncomingRequest) -> Answer:
        db_table__url, file_api_url, limit, chunk_size, memory_semaphore_size, file_server_type = DictCaster(
            Item('db_table__url', str),
            Item('file_api_url', str),
            Item('limit', int, default=10**6),
            Item('chunk_size', int, default=500),
            Item('memory_semaphore_size', int, default=500),
        ).cast_and_return(request.key_value_arguments)
        parse_result = yarl.URL(db_table__url)
        table, column = DictCaster(Item('table', str),Item('column', str)).cast_and_return(parse_result.query)
        db_url = parse_result.with_query(None).with_fragment(None)
        db_connector = DatabaseConnector(DatabaseConnector.Config(address=str(db_url)))
        was, now, total = await self.context.controller.copy_from(db_connector, table, column, file_api_url,
                                                                  limit, chunk_size, memory_semaphore_size)
        return WrappedAnswer({"was": was, "now": now, "total": total})

    def get_bucket_prefix_with_file_id(self, file_id: str) -> str:
        return f'{self.bucket_prefix}/{file_id}' if self.bucket_prefix else file_id

    @classmethod
    def get_file_id(cls, value: str) -> _FileId:
        bucket_prefix, _, id_ = value.rpartition('/')
        return cls._FileId(bucket_prefix, id_)


def generate_cache_headers(key: str, max_age: int = 60) -> dict[str, str]:
    return {"Cache-Control": f"max-age={max_age}", "ETag": f'"{key}"'}


_range_header_regex = re.compile(r"^bytes=(\d+)-(\d+)?")
_if_none_match_header_regex = re.compile(r'"(.*?[^\\])"')


def _parse_range_header(header_value: str | None) -> range:
    if header_value and (match := _range_header_regex.match(header_value)):
        start, stop = match.groups()
        return range(int(start), -1 if stop is None else int(stop))


def _parse_if_none_match_header(header_value: str | None) -> list[str]:
    return list(_if_none_match_header_regex.findall(header_value)) if header_value else []
