#  Copyright (C) 2021
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors:
#  Vasya Svintsov <v.svintsov@techokert.ru>,
#  Alexander Medvedev <a.medvedev@abm-jsc.ru>,
#  Andrey Vaydich <a.vaydich@abm-jsc.ru>

import asyncio
import io
import logging
import zipfile
from concurrent.futures import ProcessPoolExecutor
from dataclasses import dataclass
from secrets import token_hex

from file_storage.abstract_file_storage import AbstractFileStorage
from file_storage.exceptions import PathDoesNotExists
from sqlalchemy.ext.asyncio import AsyncSession

from .entities.file_metadata import FileMetadata
from .file_database_facade import FileDatabaseFacade
from .range import Range
from .thumbnail_provider import ThumbnailProvider
from .tools.file_extension import define_file_extension
from .tools.key_lock import KeyLock
from .tools.filename_preparation import prepare_filename_with_extension

logger = logging.getLogger(__name__)


class MaxArchiveBodySizeExceeded(Exception):
    pass


class MaxFilesInArchiveAmountExceeded(Exception):
    pass


class FileController:
    @dataclass
    class Config:
        max_thumbnail_creation_processes: int = 5
        max_files_in_archive: int = 100
        max_total_archive_size_bytes: int = 100 * 1024 * 1024

    @dataclass
    class Context:
        file_database_facade: FileDatabaseFacade
        file_storage: AbstractFileStorage
        thumbnail_provider: ThumbnailProvider

    def __init__(self, config: Config, context: Context) -> None:
        self._config = config
        self._context = context

        self._file_operation_lock = KeyLock()
        self._process_pool = ProcessPoolExecutor(max_workers=self._config.max_thumbnail_creation_processes)

    async def upload_file(self,
                          file_name: str,
                          file_content: bytes,
                          is_create_thumbnail: bool = False,
                          unsafe_session: AsyncSession | None = None) -> FileMetadata:
        logger.info(f'Start uploading file with file_name: {file_name}, file_size: {len(file_content)}')
        file_extension = define_file_extension(file_name, file_content)
        if file_extension is not None:
            file_name = prepare_filename_with_extension(file_name, file_extension)

        file_metadata: FileMetadata = FileMetadata.prepare(file_content, file_name, file_extension)

        async with self._file_operation_lock.restrict(file_metadata.storage_file_name), \
                self._context.file_database_facade.ensure_session(unsafe_session) as session:
            existing_file_metadata = await self._context.file_database_facade.get_info(
                session, file_metadata.storage_file_name, raise_if_not_exist=False
            )
            if existing_file_metadata is not None:
                file_metadata = existing_file_metadata
            else:
                logger.info(
                    f'File with filename: {file_metadata.full_file_name} ({file_metadata.storage_file_name}) '
                    f'does not exist. It will be saved in database')
                await self._context.file_database_facade.add(session, file_metadata)

            is_file_exist = await self._context.file_storage.check_file_existence(
                file_metadata.relative_storage_path, file_metadata.storage_file_name
            )
            if not is_file_exist:
                logger.info(f'File payload of {file_metadata.full_file_name} ({file_metadata.storage_file_name}) '
                            f'does not exist. It will be uploaded')
                await self._context.file_storage.save(
                    file_content, file_metadata.relative_storage_path, file_metadata.storage_file_name
                )
            logger.info(f'File with file_name: {file_metadata.full_file_name} ({file_metadata.storage_file_name}) '
                        f'was successfully uploaded')

            if (is_create_thumbnail and (thumbnail_maker := self._context.thumbnail_provider.get(file_extension))
                    and file_metadata.thumbnail is None):
                thumbnail_file_content = await asyncio.get_event_loop().run_in_executor(
                    self._process_pool, thumbnail_maker.process, file_content
                )
                thumbnail_file_metadata = await self.upload_file(
                    file_name, thumbnail_file_content, is_create_thumbnail=False, unsafe_session=session
                )
                thumbnail_file_metadata.parent_id = file_metadata.storage_file_name
                file_metadata.thumbnail = thumbnail_file_metadata

            return file_metadata

    async def get_file(self,
                       storage_file_name: str,
                       range_: Range | None = None,
                       unsafe_session: AsyncSession | None = None) -> tuple[FileMetadata, bytes]:
        logger.info(f'Start getting file with storage_file_name: {storage_file_name}')
        async with self._context.file_database_facade.ensure_session() as session:
            file_metadata = await self._context.file_database_facade.get_info(session, storage_file_name)

            if range_ is None:
                range_ = Range()

            file_payload = await self._context.file_storage.load(
                file_metadata.relative_storage_path, file_metadata.storage_file_name, range_.offset, range_.size,
            )
        return file_metadata, file_payload

    async def zip_files(self, storage_file_names: list[str]) -> tuple[FileMetadata, bytes]:
        logger.info(f'Start getting archived files with storage_file_names: {storage_file_names}')

        len_storage_file_names = len(storage_file_names)
        if len_storage_file_names > self._config.max_files_in_archive:
            error_msg = f'Max files amount limit exceeded: {len_storage_file_names}/{self._config.max_files_in_archive}'
            logger.error(error_msg)
            raise MaxFilesInArchiveAmountExceeded(error_msg)

        async with self._context.file_database_facade.ensure_session() as session:
            files_metadata = await self._context.file_database_facade.get_file_metadatas_by_file_names(
                session, storage_file_names
            )

        total_file_size = 0
        unique_found_file_names = set()

        for metadata in files_metadata:
            total_file_size += metadata.file_size
            unique_found_file_names.add(metadata.storage_file_name)

        if not_found_file_names := set(storage_file_names) - unique_found_file_names:
            raise PathDoesNotExists(f'Some files not found: {not_found_file_names}')

        if total_file_size > self._config.max_total_archive_size_bytes:
            error_msg = (f'Max total file size limit exceeded: '
                         f'{total_file_size}/{self._config.max_total_archive_size_bytes}')
            logger.error(error_msg)
            raise MaxArchiveBodySizeExceeded(error_msg)

        zip_buffer = io.BytesIO()

        with zipfile.ZipFile(zip_buffer, 'w', zipfile.ZIP_DEFLATED, False) as zip_file:
            for file_metadata in files_metadata:
                file_payload = await self._context.file_storage.load(
                    file_metadata.relative_storage_path, file_metadata.storage_file_name
                )
                # If file name contains '/' zipfile lib thinking that it's subdirectories,
                # splitting name by it and creating archive with subdirectories
                # e.g. 'foo/bar.pdf' - in archive will be created folder foo and inside it file bar.pdf
                if '/' in file_metadata.full_file_name:
                    file_metadata.full_file_name = file_metadata.full_file_name.replace('/', '-')
                zip_file.writestr(file_metadata.full_file_name, file_payload)

        zip_buffer.seek(0)
        file_content = zip_buffer.read()
        zip_extension = 'zip'
        file_metadata = FileMetadata.prepare(file_content, f'{token_hex(16)}.{zip_extension}', zip_extension)

        return file_metadata, file_content

    async def get_file_info(self,
                            storage_file_name: str,
                            unsafe_session: AsyncSession | None = None) -> FileMetadata:
        logger.info(f'Start getting file info with storage_file_name: {storage_file_name}')
        async with self._context.file_database_facade.ensure_session() as session:
            file_metadata = await self._context.file_database_facade.get_info(session, storage_file_name)

            is_file_exist = await self._context.file_storage.check_file_existence(
                file_metadata.relative_storage_path, file_metadata.storage_file_name
            )
            if not is_file_exist:
                raise PathDoesNotExists(storage_file_name)
        return file_metadata

    async def delete_file(self,
                          storage_file_name: str,
                          is_fake_delete: bool = False,
                          unsafe_session: AsyncSession | None = None) -> FileMetadata:
        logger.info(f'Start deleting file with storage_file_name: {storage_file_name}')
        async with self._file_operation_lock.restrict(storage_file_name), \
                self._context.file_database_facade.ensure_session() as session:
            file_metadata = await self._context.file_database_facade.get_info(session, storage_file_name)

            if is_fake_delete:
                return file_metadata

            await self._context.file_database_facade.delete(session, storage_file_name)
            await self._context.file_storage.delete(
                file_metadata.relative_storage_path, file_metadata.storage_file_name
            )
            if file_metadata.thumbnail is not None:
                await self._context.file_storage.delete(
                    file_metadata.thumbnail.relative_storage_path, file_metadata.thumbnail.storage_file_name
                )

        return file_metadata
