import dataclasses
import datetime
import functools
import inspect
import logging
import typing
from dataclasses import dataclass, field
from decimal import Decimal
from enum import Enum
from typing import Optional, Type, Any, Union

from apispec import APISpec
from dict_caster.extras import first
from http_tools import Answer
from init_helpers.dict_to_dataclass import NoValue

logger = logging.getLogger(__name__)


class ParameterLocation(str, Enum):
    query = 'query'
    path = 'path'
    header = 'header'
    cookie = 'cookie'
    body = 'body'  # INNER: not present in specification, used inside library


@dataclass(frozen=True)
class Parameter:
    name: str
    schema: Type
    in_: ParameterLocation
    body_mime_type = None
    default: Any = NoValue

    @property
    def required(self):
        return self.default is NoValue


@dataclass(frozen=True)
class QueryParameter(Parameter):
    in_: ParameterLocation = ParameterLocation.query


@dataclass(frozen=True)
class HeaderParameter(Parameter):
    in_: ParameterLocation = ParameterLocation.header


@dataclass(frozen=True)
class PathParameter(Parameter):
    in_: ParameterLocation = ParameterLocation.path


@dataclass
class Content:
    mime_type: str
    schema: Type


# @dataclass
# class Response:
#     content: Content


# T = TypeVar('T')
# 
# # @dataclass
# class JsonResponse(Generic[T]):
#     mime_type: str = "application/json"
#     def __init__(self, payload):
#         self.payload = payload

@dataclass
class RequestBody:
    content: Content


@dataclass
class SecuritySchemeType(str, Enum):
    http = "http"
    api_key = "apiKey"
    oauth2 = "oauth2"
    open_id_connect = "openIdConnect"


@dataclass(frozen=True)
class SecurityScheme:
    name: str
    type: str
    scheme: str


@dataclass(frozen=True)
class BearerSecurityScheme(SecurityScheme):
    type: str = 'http'
    scheme: str = 'bearer'


@dataclass
class Security:
    scheme: SecurityScheme
    scopes: list[str] = field(default_factory=list)


@dataclass
class Endpoint:
    path: str
    method: str
    operation_id: str
    securities: list[Security]
    parameters: list[Parameter]
    request_body: Optional[RequestBody]
    code_to_answer: dict[int, Type[Answer]]


@dataclass
class GetIdsFromXlsxResponseSchema:
    done: bool
    result: list[int]


class OpenApiWrapper:
    @dataclasses.dataclass
    class Config:
        pass

    @dataclasses.dataclass
    class Context:
        company: str
        project_group: str
        project_name: str

    def __init__(self, config: Config, context: Context):
        self.version = self._get_version()
        self.config = config
        self.context = context
        self.endpoints = []

    @staticmethod
    def _get_version() -> str:
        now = datetime.datetime.now()
        return f"{now.year - 2021}.{now.month}.{now.day}{now.hour:02d}{now.minute:02d}"

    def get_spec_dict(self) -> dict:
        spec = APISpec(
            title=f"{self.context.company}:{self.context.project_group}:{self.context.project_name}",
            version=self.version,
            openapi_version="3.0.2"
        )
        for endpoint in self.endpoints:
            self._add_endpoint_to_spec(spec, endpoint)
        return spec.to_dict()

    @staticmethod
    def _get_name_for_object(object_: Any) -> str:
        return f"component_{object_.__name__}"

    @functools.cache
    def _add_schema_to_spec(self, spec: APISpec, schema: Type, default: Any = NoValue) -> Union[str, dict]:
        logger.debug('_add_schema_to_spec: %s', schema)
        optional = False

        if getattr(schema, '__origin__', None) is typing.Union:
            type_args = getattr(schema, '__args__', None)
            non_none_types = [type_ for type_ in type_args if type_ is not type(None)]
            if len(non_none_types) != 1:
                raise TypeError(f"Got unsupported type: {schema}")
            schema = first(non_none_types)
            optional = True

        origin_type = getattr(schema, '__origin__', None)
        if origin_type is list:
            description = {
                "type": "array",
                "items": self._add_schema_to_spec(spec, getattr(schema, '__args__')[0])
            }
        elif origin_type is dict:
            description = {
                "type": "object",
                "additionalProperties": self._add_schema_to_spec(spec, getattr(schema, '__args__')[1])
            }
        elif origin_type is not None:
            raise TypeError(f"Unknown origin_type: {schema}")

        elif dataclasses.is_dataclass(schema):
            description = {
                "type": "object",
                "properties": {}
            }
            # noinspection PyDataclass
            for field_ in dataclasses.fields(schema):
                if field_.repr:
                    description["properties"][field_.name] = self._add_schema_to_spec(spec, field_.type)

            name = f"schema_{schema.__name__}"
            logger.debug('schema: %s, %s, %s', name, schema, list(description["properties"]))
            spec.components.schema(component_id=name, component=description)
            return name
        elif schema is bool:  # MUST be before int
            description = {"type": "boolean"}
        elif issubclass(schema, int):
            description = {
                "type": "integer",
                # "format": "int32", "default": 0
            }
        elif issubclass(schema, float):
            description = {
                "type": "number",
                # "format": "int32", "default": 0
            }
        elif issubclass(schema, bytes):
            description = {
                "type": "string",
                "format": "binary"
            }
        elif issubclass(schema, str):
            description = {"type": "string"}
        elif issubclass(schema, Decimal):
            description = {"type": "string"}
        else:
            raise TypeError(f"Unknown type: {schema}")

        if optional:
            description['nullable'] = optional

        if default != NoValue and not callable(default) and not inspect.isawaitable(default):
            description['default'] = default

        return description

    @functools.cache
    def _add_answer_to_spec(self, spec: APISpec, answer: Type[Answer]) -> str:
        content_type = answer.get_class_content_type()
        schema = answer.get_class_payload_type()
        name = answer.__name__
        # response_schema = response.__args__[0]
        # response_type = response.__origin__
        # name = f"response_{response_type.__name__}[{response_schema.__name__}]"
        schema_name_or_description = self._add_schema_to_spec(spec, schema)

        spec.components.response(
            component_id=name, component={
                "description": "",
                "content": {
                    content_type: {"schema": schema_name_or_description}
                }
            }
        )
        return name

    @functools.cache
    def _add_security_schema_to_spec(self, spec: APISpec, security_schema: SecurityScheme) -> str:
        spec.components.security_scheme(
            component_id=security_schema.name,
            component={
                "description": "",
                "type": security_schema.type,
                "scheme": security_schema.scheme,
            })
        return security_schema.name

    def _add_endpoint_to_spec(self, spec: APISpec, endpoint: Endpoint):
        responses: dict[str, str] = {}
        securities: list[dict] = []
        parameters: list[dict] = []
        operation_dict = {
            "summary": "",
            "operationId": endpoint.operation_id,
            "security": securities,
            "parameters": parameters,
            "responses": responses,
        }

        if endpoint.request_body is not None:
            schema_name = self._add_schema_to_spec(spec, endpoint.request_body.content.schema)
            operation_dict["requestBody"] = {
                "required": True,
                "content": {
                    endpoint.request_body.content.mime_type: {"schema": schema_name}
                }
            }

        for parameter in endpoint.parameters:
            schema_name = self._add_schema_to_spec(spec, parameter.schema, default=parameter.default)
            parameters.append(
                {"name": parameter.name, "in": parameter.in_, "required": parameter.required, "schema": schema_name}
            )

        for security in endpoint.securities:
            security_scheme_name = self._add_security_schema_to_spec(spec, security.scheme)
            securities.append({security_scheme_name: security.scopes})

        for code, answer in endpoint.code_to_answer.items():
            response_name = self._add_answer_to_spec(spec, answer)
            responses[str(int(code))] = response_name

        spec.path(endpoint.path, operations={endpoint.method: operation_dict})

    def register_endpoint(self, endpoint: Endpoint):
        self.endpoints.append(endpoint)
