#  Copyright (C) 2023
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Mike Orlov <m.orlov@abm-jsc.ru>
from dataclasses import dataclass

import entity_read.sql.inter.other_func
import typed_json.typed_In_dumps
from entity_read import sql
from entity_read.entity import Entity

from .expression import Expression


key_to_function_factory = {
    "len": entity_read.sql.inter.other_func.FunctionGetSubqueryLength,
    "exists": entity_read.sql.inter.other_func.FunctionExists,
    "not_exists": entity_read.sql.inter.other_func.FunctionNotExists,
    "to_str": entity_read.sql.inter.other_func.FunctionCastToStr,
    "sub": entity_read.sql.inter.other_func.FunctionSub,
    "div": entity_read.sql.inter.other_func.FunctionDiv,
}


@dataclass(frozen=True, repr=False)
class Function(Expression):
    key: str
    args: list[Expression]

    def __repr_in_dumps__(self) -> str | dict:
        self_dict_repr: dict = super().__repr_in_dumps__()
        if isinstance(self_dict_repr, dict):
            arg_reprs = [arg.__repr_in_dumps__() for arg in self_dict_repr["args"]]
            if all(isinstance(arg_repr, str) for arg_repr in arg_reprs):
                return f'func.{self.key}({",".join(arg_reprs)})'
            self_dict_repr['key'] = f"'{self_dict_repr['key']}'"
            self_dict_repr['args'] = arg_reprs
            self_dict_repr[typed_json.typed_In_dumps.TYPE_KEY] = self.get_prefix() + Function.__name__
        return self_dict_repr

    @property
    def keep_subquery(self) -> bool:
        return self._get_factory().keep_subquery

    def _get_factory(self) -> type[sql.atoms.Function]:
        if (function_factory := key_to_function_factory.get(self.key)) is None:
            raise KeyError(f"Not found function {self.key!r}")
        return function_factory

    def eval(self, entity_type: type[Entity], variables: dict[str, sql.atoms.Selectable]) -> sql.atoms.Function:
        function_factory = self._get_factory()
        # noinspection PyArgumentList
        return function_factory(args=[arg.eval(entity_type, variables) for arg in self.args])

    def shortcut(self) -> str:
        return f"func.{self.key}({','.join([arg.shortcut() for arg in self.args])})"


class Exists(Function):
    def __init__(self, selectable: Expression):
        super().__init__(key="exists", args=[selectable])

    def __repr_in_dumps__(self) -> str | dict:
        from .subquery import SubQuery
        self_dict_repr: str | dict = super().__repr_in_dumps__()
        if isinstance(self_dict_repr, str):
            assert len(self.args) == 1, 'Expected exactly one argument'
            if isinstance(self.args[0], SubQuery):
                return self.args[0].__repr_in_dumps__().replace('subquery', 'exists', 1)
        return self_dict_repr
