#  Copyright (C) 2024
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Mikhail Mamrov <m.mamrov@abm-jsc.ru>
#
import logging
from dataclasses import dataclass
from enum import StrEnum, unique

from aiohttp import hdrs
from http_tools import AbmServiceConnector
from init_helpers.dict_to_dataclass import dict_to_dataclass

logger = logging.getLogger(__name__)


class TokenId(str):
    pass


@unique
class TokenType(StrEnum):
    access_token = "access_token"
    refresh_token = "refresh_token"


@dataclass
class PortalPayload:
    id: int
    type_id: int
    origin: str
    language_id: int
    trial_ended_at: int | None


@dataclass
class UserVerificationPayload:
    verified_at: int


@dataclass
class UserStatusBindingPayload:
    status_id: int


@dataclass
class PortalMemberPayload:
    id: int
    role_id: int
    first_name: str
    last_name: str
    nickname: str | None
    email: str
    avatar_id: str | None
    user_onboarding_step: str
    business_model: str | None
    registered_at: int
    tutorial_status: bool
    inactive_window_status: bool
    verification: UserVerificationPayload | None
    statuses: list[UserStatusBindingPayload]


@dataclass(frozen=True)
class JWTPayload:
    portal: PortalPayload
    portal_member: PortalMemberPayload
    id: TokenId
    type: TokenType
    iat: int
    exp: int


class AuthServerConnector:
    Config = AbmServiceConnector.Config
    Context = AbmServiceConnector.Context

    def __init__(self, config: Config, context: Context) -> None:
        self.config = config
        self.context = context
        self._connector = AbmServiceConnector(config, context)

    async def get_access_token_payload(self, token: str, origin: str) -> JWTPayload:
        answer = await self._connector.post(
            path="v1/access_token/payload/get",
            payload={"access_token": token},
            headers={hdrs.ORIGIN: origin},
        )
        return dict_to_dataclass(answer, JWTPayload)

    async def verify_access_token(self, token: str, origin: str) -> None:
        await self._connector.post(
            path="v1/auth/access_token/verify",
            payload={"access_token": token},
            headers={hdrs.ORIGIN: origin},
        )
