#  Copyright (C) 2024
#  ABM JSC, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Albakov Ruslan <r.albakov@abm-jsc.ru>
import asyncio
import logging
import time
from contextlib import asynccontextmanager
from dataclasses import dataclass
from typing import Hashable

from .redis_connector import RedisConnector
from .abstract_semaphore import AbstractSemaphore

logger = logging.getLogger(__name__)

class RedisSemaphore(AbstractSemaphore):
    @dataclass
    class Context:
        redis: RedisConnector

    @dataclass
    class Config:
        key_time_to_live_s: int = 120
        sleep_time_s: int = 3

    def __init__(self, config: Config, context: Context) -> None:
        self.config = config
        self.context = context

    @asynccontextmanager
    async def restrict(self, key: Hashable, concurrency_limit: int = 1) -> None:
        """
        This method limits the number of connections by ``concurrency_limit``.
        If resources are free, the ``key`` is set to a hash in Redis.  \
        If all resources are locked, the program waits until the resource is released.
        :param key: Hashable, set field in the hash stored at key in Redis.
        :param concurrency_limit: int, the number of restrict connections.
        :return: None
        """
        current_time = str(time.time())
        redis_hash = str(key)
        redis_key = f'{redis_hash}_{current_time}'
        await self._check_limit_connection(redis_hash, redis_key, concurrency_limit)
        logger.debug(f"{redis_key} connection acquired")
        try:
            yield
        finally:
            logger.debug(f"{redis_key} connection released")
            await self.context.redis.hdel(redis_hash, redis_key)

    async def _check_limit_connection(self, redis_hash: str, redis_key: str, concurrency_limit: int) -> None:
        """
        This method checked block.

        The number of connections cannot be more than the set restrict ``concurrency_limit``.

        If the resources are free, key ``redis_key`` is set to hash ``redis_hash``.  \
        If all resources are blocked, the program waits until the resource is released.
        :param redis_hash: str, the name of the hash in Redis.
        :param redis_key: str, set field in the hash stored at key in Redis.
        :param concurrency_limit: int, the number of restrict connections.
        :return: None.
        """
        while await self.context.redis.hset_and_get_hlen(
                redis_hash, redis_key, seconds=self.config.key_time_to_live_s
        ) > concurrency_limit:
            await self.context.redis.hdel(redis_hash, redis_key)
            await asyncio.sleep(self.config.sleep_time_s)
