#  Copyright (C) 2024
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors:
#  Orlov Mikhail <m.orlov@abm.jsc.ru>,

import logging
import os
import typing
from dataclasses import dataclass

# from sqlalchemy import select, exists
from sqlalchemy import select, exists
# from sqlalchemy.orm import joinedload

from .database import Database, WrappedSession
from .extra import AnySession, assert_raise
from .entities.link_info import LinkInfo
from .entities.file_info import FileInfo
from .file_keeper import FileKeeper
from .tools.file_extension import define_file_extension
from .tools.key_lock import KeyLock

logger = logging.getLogger(__name__)
FI = typing.TypeVar('FI', bound=FileInfo)
LI = typing.TypeVar('LI', bound=LinkInfo)


class LinkKeeper(typing.Generic[LI]):
    @dataclass
    class Config:
        pass

    @dataclass
    class Context:
        file_keeper: FileKeeper
        database: Database

    def __init__(self, context: Context, config: Config = None, link_class: type[LI] = LinkInfo,
                 id_generator: typing.Callable[[], str] | None = None) -> None:
        self.config = config or self.Config()
        self.context = context
        self.link_class = link_class
        self.id_generator = id_generator or self.default_id_generator

    @staticmethod
    def default_id_generator() -> str:
        return os.urandom(8).hex()

    async def _get_link_info(self, link_id: str, session: AnySession) -> LI | None:
        # noinspection PyTypeChecker
        query = select(self.link_class).where(self.link_class.id == link_id)
        # query = query.options(joinedload(self.link_class)) if with_file_info else query
        async with self.context.database.ensure_session(session) as session:
            return await session.scalar_or_none(query)

    async def _get_link_infos(self, link_ids: list[str], session: AnySession) -> list[LI]:
        async with self.context.database.ensure_session(session) as session:
            step = 32760
            result = []
            for i in range(0, len(link_ids), step):
                chunk = link_ids[i*step: (i+1)*step]
                # noinspection PyTypeChecker,PyUnresolvedReferences
                result += await session.scalars(select(self.link_class).where(self.link_class.id.in_(chunk)))
            return list(result)
            # return await session.scalars(select(self.link_class).where(self.link_class.id.in_(link_ids)))

    async def _check_exists_links_with_file_info_key(self, session: AnySession, file_info_key: str) -> bool:
        async with self.context.database.ensure_session(session) as session:
            # noinspection PyTypeChecker
            return await session.scalar(select(exists().where(self.link_class.file_info_key == file_info_key)))

    async def upload(self, content: bytes, filename: str = '', link_id: str | None = None, session: AnySession = None,
                     link_kwargs: dict | None = None, file_kwargs: dict | None = None) -> LI:
        logger.info(f'Uploading link {filename=}, size: {len(content)} bytes')
        async with self.context.database.ensure_session(session) as session:
            file_info = await self.context.file_keeper.upload(content, session, file_kwargs)
            link_id = link_id or self.id_generator()
            await session.add_and_flush(link_info := self.link_class(
                id=link_id, file_info_key=file_info.key, filename=filename,
                extension=define_file_extension(filename, content), **(link_kwargs or {})
            ))
            return await self.head(link_id, session)
        # return link_info

    async def get(self, link_id: str, byte_range: range | None = None,
                  session: AnySession = None) -> tuple[LI, bytes] | None:
        logger.info(f'Getting link with {link_id=}')
        link_info = await self._get_link_info(link_id=link_id, session=session)
        assert_raise(link_info, FileNotFoundError, link_id)
        # link_info = await self._get_link_info(link_id=link_id, session=session, with_file_info=True)
        file_content = await self.context.file_keeper.read(link_info.file_info_key, byte_range)
        return link_info, file_content

    async def get_multiple(self, link_ids: list[str], session: AnySession = None) -> list[bytes]:
        logger.info(f'Getting links with {link_ids=}')
        link_infos = await self._get_link_infos(link_ids=link_ids, session=session)
        return [await self.context.file_keeper.read(link.key) for link in link_infos]

    async def read(self, file_info_key: str, byte_range: range | None = None) -> bytes:
        logger.debug(f'Reading bytes by {file_info_key=}')
        return await self.context.file_keeper.read(file_info_key, byte_range)

    async def head(self, link_id: str, session: AnySession = None) -> LI | None:
        logger.info(f'Getting link info with {link_id=}')
        return await self._get_link_info(link_id=link_id, session=session)
        # return await self._get_link_info(link_id=link_id, session=session, with_file_info=True)

    async def head_multiple(self, link_ids: list[str], session: AnySession = None) -> list[LI]:
        logger.info(f'Getting link infos with {link_ids=}')
        return await self._get_link_infos(link_ids=link_ids, session=session)

    async def list(self, link_ids: list[str], session: AnySession = None) -> list[str]:
        logger.info(f'Listing {len(link_ids)} links')
        links = await self._get_link_infos(link_ids=link_ids, session=session)
        return [link.id for link in links]

    async def delete(self, link_id: str, delete_orphan_file: bool = True, session: AnySession = None) -> LI:
        logger.info(f'Deleting link: {link_id=}')
        async with self.context.database.ensure_session(session) as session:
            link_info = await self._get_link_info(link_id=link_id, session=session)
            # link_info = await self._get_link_info(link_id=link_id, session=session, with_file_info=False)
            session.delete_and_flush(link_info)
            if delete_orphan_file:
                if not await self._check_exists_links_with_file_info_key(session, link_info.file_info_key):
                    logger.info(f"Not found any links pointing to {link_info.file_info_key}, delete orphan file")
                    await self.context.file_keeper.delete(link_info.storage_file_name)
        return link_info
