import logging
from dataclasses import dataclass, field
from typing import Type

from dict_caster.extras import to_list

from ..elements.abstract.sqlable import Sqlable
from ..elements.order import Order
from ..elements.filter_by import Condition
from ..elements.output_format import OutputFormat
from ..elements.result import SelectResult
from ..elements.table import Table
from .abstract_statement import AbstractStatement
from ..elements.abstract.selectable import Selectable

logger = logging.getLogger(__name__)


@dataclass
class Select(AbstractStatement):
    table: Type[Table]
    selectables: list[Selectable] | Selectable | None = field(default_factory=list)
    condition: Condition.get_condition_type() | None = None
    group_by: list[Sqlable] | Sqlable | None = field(default_factory=list)
    order_by: list[Order] | Order | None = field(default_factory=list)
    offset: int | None = None
    limit: int | None = None
    output_format: OutputFormat = OutputFormat.Dict

    def __post_init__(self):
        self.table.get_table_engine().preprocess_select_request(self)

        self.selectables = to_list(self.selectables) if self.selectables is not None else []
        self.group_by = to_list(self.group_by) if self.group_by is not None else []
        self.order_by = to_list(self.order_by) if self.order_by is not None else []

    def generate_sql(self):
        table_name = self.table.get_name()
        select_sql = ', '.join(selectable.to_selector() for selectable in self.selectables)
        where_sql = f'\nWHERE {self.condition.to_sql()}' if self.condition else ''

        group_by_part = ', '.join((g.to_sql() for g in self.group_by))
        group_sql = f'\nGROUP BY {group_by_part}' if group_by_part else ''

        order_by_part = ', '.join((o.to_sql() for o in self.order_by))
        order_sql = f'\nORDER BY {order_by_part}' if order_by_part else ''

        limit_sql = f'\nLIMIT {self.limit}' if self.limit else ''
        offset_sql = f'\nOFFSET {self.offset}' if self.offset else ''
        sql = f"SELECT {select_sql} FROM {table_name} {where_sql} {group_sql} {order_sql} {limit_sql} {offset_sql}"
        logger.debug(f"constructed {sql=}")
        return sql

    def form_result(self, payload: str) -> SelectResult:
        return SelectResult(payload, self.selectables, self.output_format)
