#  Copyright (C) 2023
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Vasya Svintsov <v.svintsov@techokert.ru>
import yarl
from aiohttp import ClientSession, ClientResponse
from aiohttp.typedefs import StrOrURL

from ..metrics_registry import MetricsRegistry
from ..request_id_exemplar import get_request_id_exemplar


class MonitoredClientSession(ClientSession):
    def __init__(self, metrics: MetricsRegistry, *args, **kwargs) -> None:
        self._metrics = metrics
        super().__init__(*args, **kwargs)

    async def _request(
            self, method: str, str_or_url: StrOrURL, extended_metrics_labels: dict[str, str] | None = None, **kwargs
    ) -> ClientResponse:
        exemplar = get_request_id_exemplar()
        labels = {'method': method.upper(), 'url': str(yarl.URL(str_or_url).with_query({}))}
        labels |= extended_metrics_labels or {}
        self._metrics.counter('http_client__requests', labels).inc(exemplar=exemplar)
        with self._metrics.track('http_client__requests', labels, exemplar=exemplar) as tracker:
            try:
                response = await super()._request(method, str_or_url, **kwargs)
                tracker.labels['status_code'] = response.status
            except Exception as er:
                tracker.labels['error'] = type(er).__name__
                raise
        self._metrics.counter('http_client__input_traffic', labels).inc(response.content_length or 0, exemplar=exemplar)
        return response
