#  Copyright (C) 2021
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Vasya Svintsov <v.svintsov@techokert.ru>, Alexander Medvedev <a.medvedev@abm-jsc.ru>

import urllib.parse
from dataclasses import dataclass, field
from typing import TypeVar, Optional, Union

from http_tools.mime_types import ContentType

from .entities.file_metadata import FileMetadata
from .tools.file_extension import define_file_mime_type_by_extension


TFileDescriptor = TypeVar('TFileDescriptor', bound='FileDescriptor')


@dataclass
class FDescriptor:
    class NotLoad:
        pass

    id: str
    size: int
    full_file_name: str
    file_extension: Optional[str] = None  # openapi-tools-abm not support 'str | None'
    content_type: str = field(init=False)
    quoted_full_file_name: str = field(init=False)
    bytes_content: Union[bytes, NotLoad] = field(repr=False, default_factory=NotLoad)

    def __post_init__(self) -> None:
        self.content_type = ContentType.Octet.value
        if (self.file_extension is not None) and (mime_type := define_file_mime_type_by_extension(self.file_extension)):
            self.content_type = mime_type

        self.quoted_full_file_name = urllib.parse.quote(self.full_file_name, ' ').replace(' ', '_')


@dataclass
class FileDescriptor(FDescriptor):
    thumbnail: Optional[FDescriptor] = None  # openapi-tools-abm not support TypeVar


def get_file_descriptor_from_file_metadata(file_metadata: FileMetadata,
                                           file_payload: bytes | None = None) -> FileDescriptor:
    thumbnail = None
    if file_metadata.thumbnail is not None:
        thumbnail = FileDescriptor(
            id=file_metadata.thumbnail.storage_file_name,
            full_file_name=file_metadata.thumbnail.full_file_name,
            size=file_metadata.thumbnail.file_size,
            file_extension=file_metadata.thumbnail.file_extension,
            bytes_content=file_payload if file_payload is not None else FileDescriptor.NotLoad(),
        )
    return FileDescriptor(id=file_metadata.storage_file_name,
                          full_file_name=file_metadata.full_file_name,
                          size=file_metadata.file_size,
                          file_extension=file_metadata.file_extension,
                          thumbnail=thumbnail,
                          bytes_content=file_payload if file_payload is not None else FileDescriptor.NotLoad(),
                          )
