#  Copyright (C) 2021
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Vasya Svintsov <v.svintsov@techokert.ru>

import logging
from dataclasses import dataclass

from async_tools import AsyncOnStart
from ..abstract_file_storage import AbstractFileStorage
from ..exceptions import BucketAlreadyExist
from ..s3_file_storage.s3_interface import S3Interface, Range

logger = logging.getLogger(__name__)


class S3FileStorage(AbstractFileStorage, AsyncOnStart):
    @dataclass
    class Config(S3Interface.Config):
        bucket_name: str = "abm-storage"

    def __init__(self, config: Config):
        logger.info(f"{type(self).__name__} init. Config: {config}")
        self._s3_interface = S3Interface(config)
        self._bucket_name = config.bucket_name

    async def _on_start(self) -> None:
        try:
            await self._s3_interface.create_bucket(self._bucket_name)
        except BucketAlreadyExist:
            logger.info(f"Bucket already exist. bucket_name = {self._bucket_name}")
        else:
            logger.info(f"Bucket created. bucket_name = {self._bucket_name}")

    async def save(self, content: bytes, relative_path: str, name: str, allow_rewrite: bool = False) -> None:
        await self._s3_interface.upload_file(
            bucket_name=self._bucket_name, payload=content, key=f"{relative_path}/{name}", allow_rewrite=allow_rewrite)

    async def load(self, relative_path: str, name: str, offset: int = 0, size: int = -1) -> bytes:
        return await self._s3_interface.get_file(
            bucket_name=self._bucket_name, key=f"{relative_path}/{name}", range_=Range(offset, size)
        )

    async def delete(self, relative_path: str, name: str) -> None:
        return await self._s3_interface.delete_file(bucket_name=self._bucket_name, keys=f"{relative_path}/{name}")

    async def check_file_existence(self, relative_path: str, name: str) -> bool:
        return await self._s3_interface.check_file_existence(
            bucket_name=self._bucket_name, key=f"{relative_path}/{name}")
