#  Copyright (C) 2021
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Vasya Svintsov <v.svintsov@techokert.ru>

import logging
import os
from concurrent.futures import ProcessPoolExecutor
from dataclasses import dataclass
from pathlib import Path

import aiofiles
from aiofile import AIOFile, Writer
from aiofiles.os import remove
from aiofiles.ospath import exists

from ..abstract_file_storage import AbstractFileStorage
from ..exceptions import FileAlreadyExists, PathDoesNotExists, InsecurePath

logger = logging.getLogger(__name__)


class DiscFileStorage(AbstractFileStorage):
    @dataclass
    class Config:
        location: str
        max_workers: int | None = None

    def __init__(self, config: Config):
        self.config = config
        self._process_pool_executor = ProcessPoolExecutor(max_workers=self.config.max_workers)
        logger.info(f"{type(self).__name__} inited. Location: {self.config.location}")

    async def save(self, content: bytes, relative_path: str, name: str, allow_rewrite: bool = False) -> None:
        dir_path = f"{self.config.location}/{relative_path}"
        if not await aiofiles.ospath.exists(dir_path, executor=self._process_pool_executor):
            logger.info(f"Path = {dir_path} does not exists. Will be created")
            await aiofiles.os.makedirs(dir_path)

        full_path = f"{dir_path}/{name}"
        logger.info(f"Full path = {full_path}")
        if Path(full_path).exists() and not allow_rewrite:
            raise FileAlreadyExists(full_path)

        logger.info(f"Saving file along the path: {full_path}")
        async with AIOFile(full_path, 'wb', executor=self._process_pool_executor) as file:
            writer = Writer(file)
            await writer(content)

    async def load(self, relative_path: str, name: str, offset: int = 0, size: int = -1) -> bytes:
        dir_path = f"{self.config.location}/{relative_path}"
        full_path = f"{dir_path}/{name}"
        if not Path(full_path).exists():
            raise PathDoesNotExists(full_path)

        if not os.path.realpath(full_path).startswith(os.path.realpath(dir_path)):
            raise InsecurePath(full_path)

        logger.info(f"Loading file along the path: {full_path}")
        async with AIOFile(full_path, 'rb', executor=self._process_pool_executor) as file:
            content = await file.read(size, offset)

        return content

    async def delete(self, relative_path: str, name: str) -> None:
        dir_path = f"{self.config.location}/{relative_path}"
        full_path = f"{dir_path}/{name}"

        if not Path(full_path).exists():
            raise PathDoesNotExists(full_path)

        logger.info(f"Deleting file along the path: {full_path}")
        await aiofiles.os.remove(full_path, executor=self._process_pool_executor)
        if not await aiofiles.os.listdir(dir_path, executor=self._process_pool_executor):
            await aiofiles.os.rmdir(dir_path, executor=self._process_pool_executor)

    async def check_file_existence(self, relative_path: str, name: str) -> bool:
        full_path = f"{self.config.location}/{relative_path}/{name}"
        return Path(full_path).exists()
