import logging
from typing import Any, Union,  Optional

from http_tools.http_server_connector import HttpServerConnector

from .filters import EntityFilter

logger = logging.getLogger(__name__)


class EntityServerConnector:
    Config = HttpServerConnector.Config
    Context = HttpServerConnector.Context

    def __init__(self, config: Config, context: Context) -> None:
        self._http_connector = HttpServerConnector(config, context)

    async def _get_entity(self,
                          entity_name: str,
                          columns: list[str] = None,
                          filter_by: Optional[list[EntityFilter]] = None,
                          search_by: Optional[list[EntityFilter]] = None,
                          limit: Optional[int] = 10 ** 9,
                          offset: Optional[int] = None) -> dict:
        filter_by = [{"attribute": f"{x.column}", "operator": f"{x.operator}", "value": x.value}
                     for x in filter_by] if filter_by else []

        search_by = [{"attribute": f"{x.column}", "operator": f"{x.operator}", "value": x.value}
                     for x in search_by] if search_by else []
        json_payload = {"filter_by": filter_by, "search_by": search_by}

        if columns is not None:
            json_payload["columns"] = columns

        if limit:
            json_payload["limit"] = limit

        if offset:
            json_payload["offset"] = limit

        return await self._call('get', entity_name, json_payload)

    async def _upsert_entity(self, entity_name: str, values: Union[list, dict]) -> dict:
        json_payload = {"values": values}
        return await self._call('upsert', entity_name, json_payload)

    async def _call(self, method_name: str, entity_name: str, json_payload: Any) -> dict:
        address = f"/entity/{entity_name}/{method_name}"
        logger.debug(f"making request: {address} with: {json_payload}")
        result = await self._http_connector.post_json(address, headers=self._construct_headers(), payload=json_payload)
        answer = result['result']
        return answer

    def _construct_headers(self) -> dict[str, str]:
        headers = {"Connection": "Keep-Alive"}
        return headers
