import uuid
from contextvars import ContextVar
from typing import Final, Optional, Awaitable

import aiohttp
from aiohttp import web


X_REQUEST_ID: Final[str] = 'X-Request-ID'
CONTEXT_VAR_REQUEST_ID: ContextVar[Optional[str]] = ContextVar(X_REQUEST_ID, default=None)


def generate_random_request_id() -> str:
    return str(uuid.uuid4())


@web.middleware
async def request_tracing_middleware(
        request: web.Request, handler: aiohttp.typedefs.Handler, *args, **kwargs
) -> web.StreamResponse | Awaitable[web.StreamResponse]:
    request_id = request.headers.get(X_REQUEST_ID, generate_random_request_id())
    CONTEXT_VAR_REQUEST_ID.set(request_id)
    response = await handler(request)
    response.headers[X_REQUEST_ID] = request_id
    return response
