import argparse
import asyncio
import os
from dataclasses import dataclass

import init_helpers
from async_tools import AsyncInitable, AsyncDeinitable
from file_storage.abstract_file_storage import AbstractFileStorage
from file_storage.get_file_storage import get_file_storage
from http_tools import HttpServer
from sqlalchemy_tools.database_connector.abstract_database_connector import AbstractDatabaseConnector
from sqlalchemy_tools.database_connector.database_connector import DatabaseConnector

from .controller import Controller
from .database import Database
from .thumbnail.thumbnail_provider import ThumbnailProvider
from .http_handler import HttpHandler
from .thumbnail.image_thumbnail_procressor import ImageThumbnailProcessor

init_helpers.init_logs(init_helpers.LogsConfig())


def get_var(key: str) -> str:
    parser = argparse.ArgumentParser()
    parser.add_argument(f'--{key}')
    parsed_and_other: tuple[argparse.Namespace, list[str]] = parser.parse_known_args()
    parsed, _ = parsed_and_other
    if result := getattr(parsed, key, None):
        return result
    return os.environ.get(key)


async def main():
    assert (listen := get_var('listen')), 'Missing "listen", pass it in env variable or as call arg: --listen=<db_url>'
    assert (db_url := get_var('db_url')), 'Missing db_url, pass it in env variable or as call arg: --db_url=<db_url>'
    assert (file_storage_url := get_var('file_storage_url')), \
        'Missing file_storage_url, pass it in env variable or as call arg: --file_storage_url=<file_storage_url>'

    @dataclass
    class Context(AsyncInitable, AsyncDeinitable):
        http_server: HttpServer = None
        database_connector: AbstractDatabaseConnector = None
        database: Database = None
        file_storage: AbstractFileStorage = None
        thumbnail_provider: ThumbnailProvider = None
        controller: Controller = None

        def __post_init__(self) -> None:
            AsyncInitable.__init__(self)
            AsyncDeinitable.__init__(self)

    context = Context()
    interface, port = listen.split(":", 1)
    context.http_server = HttpServer(HttpServer.Config(interface=interface, port=port),
                                     HttpServer.Context(instance_id="module"))
    context.database_connector = DatabaseConnector(DatabaseConnector.Config(address=db_url))
    context.database = Database(context)
    context.file_storage = get_file_storage(url=file_storage_url)
    context.thumbnail_provider = ThumbnailProvider(ThumbnailProvider.Config(),
                                                   ImageThumbnailProcessor(ImageThumbnailProcessor.Config()))
    context.controller = Controller(Controller.Config(), context)
    context.http_handler = HttpHandler(context)

    await context.async_init()
    print(f"Init done, listen on {listen}")
    await asyncio.sleep(10 ** 10)


asyncio.run(main())
