#  Copyright (C) 2021
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Vasya Svintsov <v.svintsov@techokert.ru>, Alexander Medvedev <a.medvedev@abm-jsc.ru>
import traceback
from functools import wraps
from typing import Union, Callable, Awaitable

from dict_caster import ItemException
from file_storage.exceptions import PathDoesNotExists
from http_tools import ErrorAnswer, Answer, HttpStatusCode
from http_tools.request import IncomingRequest
from sqlalchemy.exc import SQLAlchemyError
from botocore.exceptions import ConnectionError

# from .database import FileMetadataDoesNotExists


Handler = Union[Callable[[IncomingRequest], Awaitable[Answer]], Callable[[IncomingRequest], Answer]]


def error_middleware(handler: Handler):
    @wraps(handler)
    async def wrapper(*args, **kwargs) -> Answer:
        try:
            return await handler(*args, **kwargs)
        except (LookupError, FileNotFoundError, PathDoesNotExists) as er:
            traceback.print_exc()
            return ErrorAnswer(er, status=HttpStatusCode.NotFound)
        except (ItemException, KeyError, ValueError) as er:
            traceback.print_exc()
            return ErrorAnswer(repr(er), status=HttpStatusCode.UnprocessableEntity)
        except (ConnectionError, SQLAlchemyError) as er:
            traceback.print_exc()
            return ErrorAnswer(repr(er), status=HttpStatusCode.BadGateway)
    return wrapper
