#  Copyright (C) 2021
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Mike Orlov <m.orlov@abm-jsc.ru>
#
import dataclasses
from dataclasses import dataclass
from typing import List, Union, Type, Any, Optional

from .dict_to_dataclass import get_dataclass_field_name_to_field


@dataclass
class FieldHint:
    name: str
    type: type
    default: Any = None


@dataclass
class StructureHint:
    fields: List[Union[FieldHint, 'StructureHint']]
    name: Optional[str] = None

    @staticmethod
    def from_dataclass(dataclass_type: type) -> 'StructureHint':
        return dataclass_to_hint(dataclass_type)

    def to_ini(self):
        return hint_to_ini(self)


def get_name_from_type(type_) -> str:
    type_origin = getattr(type_, '__origin__', None)
    type_args = getattr(type_, '__args__', None)
    if type_origin and type_args:
        origin_type = getattr(type_, '__origin__')
        inner_types = getattr(type_, '__args__')
        result = get_name_from_type(origin_type)
        if inner_types:
            result += '[' + (', '.join(list(get_name_from_type(sub_type) for sub_type in inner_types))) + ']'
        return result
    elif isinstance(type_, dataclasses.InitVar):
        return get_name_from_type(type_.type)
    elif type_ is Union:
        return "Union"
    return type_.__name__


def hint_to_ini(hint: StructureHint) -> str:
    res = ""
    for f in hint.fields:
        if res:
            res += "\n"
        if isinstance(f, FieldHint):
            default = ""
            if f.default != dataclasses.MISSING:
                default = ", default = " + str(f.default)
            res += f'{f.name} = {get_name_from_type(f.type)}{default}'
        elif isinstance(f, StructureHint):
            res += f'[{f.name}]\n{hint_to_ini(f)}\n'
        else:
            raise TypeError(f"unexpected type: {type(f)}")
    return res


def dataclass_to_hint(dataclass_type: Type) -> StructureHint:
    if not dataclasses.is_dataclass(dataclass_type):
        raise TypeError("expected dataclass type")

    args = []
    dataclass_fields = get_dataclass_field_name_to_field(dataclass_type)
    for field_name in dataclass_fields:
        field = dataclass_fields[field_name]
        if not field.init:
            continue

        default = dataclasses.MISSING
        if field.default != dataclasses.MISSING:
            default = field.default
        elif field.default_factory != dataclasses.MISSING:
            default = field.default_factory

        field_type = field.type
        if dataclasses.is_dataclass(field_type):
            hint = dataclass_to_hint(field_type)
            hint.name = field_name
        else:
            hint = FieldHint(field_name, field_type, default)

        args.append(hint)
    return StructureHint(args)
