#  Copyright (C) 2024
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Mike Orlov <m.orlov@abm-jsc.ru>
import operator
from typing import ClassVar

from run_markdown.base_comparator import BaseComparator
from run_markdown.comparison_buffer import ComparisonBuffer
from run_markdown.comparison_line import TextComparisonLine
from run_markdown.utils import compare_string_with_pattern


class TextComparator(BaseComparator):
    chars_to_ignore: ClassVar[str] = '\r'

    def compare(self, value: str) -> ComparisonBuffer:
        expected = self.expected
        actual = value
        for char in self.chars_to_ignore:
            expected = expected.replace(char, '')
            actual = actual.replace(char, '')
        expected_lines = expected.splitlines()
        actual_lines = actual.splitlines()
        return self._compare(expected_lines, actual_lines)

    def _compare(
            self,
            expected_lines: list[str], actual_lines: list[str],
            expected_lines_shift: int = 0, actual_lines_shift: int = 0
    ) -> ComparisonBuffer:
        result = ComparisonBuffer()
        expected_line_i = expected_lines_shift
        actual_line_i = actual_lines_shift
        while expected_line_i < len(expected_lines) and actual_line_i < len(actual_lines):
            expected_line = expected_lines[expected_line_i]
            actual_line = actual_lines[actual_line_i]
            if expected_line == '***':
                if expected_line_i + 1 == len(expected_lines):
                    shift = len(actual_lines) - actual_line_i
                    buffer = ComparisonBuffer()
                else:
                    shift_to_buffer = {}
                    for i in range(len(actual_lines) - actual_line_i):
                        shift_to_buffer[i] = self._compare(
                            expected_lines, actual_lines, expected_line_i + 1, actual_line_i + i)
                    shift, buffer = next(iter(sorted(shift_to_buffer.items(), key=lambda x: x[1].exceptions_amount)))
                    del shift_to_buffer
                for i in range(shift):
                    actual_line = actual_lines[actual_line_i]
                    result.append(TextComparisonLine(expected_line, actual_line))
                    actual_line_i += 1
                result.values.extend(buffer.values)
                return result
            elif '...' in expected_line:
                result.append(TextComparisonLine(
                    expected_line, actual_line,
                    None if compare_string_with_pattern(expected_line, actual_line) else ValueError('Value mismatch')))
            else:
                result.append(TextComparisonLine(
                    expected_line, actual_line,
                    None if expected_line == actual_line else ValueError('Value non equal')))

            expected_line_i += 1
            actual_line_i += 1

        for i in range(actual_line_i, len(actual_lines)):
            result.append(TextComparisonLine('', actual_lines[i], ValueError(f"Extra line")))
        for i in range(expected_line_i, len(expected_lines)):
            if (expected_line := expected_lines[i]) != '***':
                result.append(TextComparisonLine(expected_line, '', ValueError(f"Missing line")))
        return result
