#  Copyright (C) 2023
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Vasiliev Ivan <i.vasiliev@technokert.ru>


import logging
from dataclasses import dataclass
from typing import Optional
from http_tools.http_server_connector import HttpServerConnector


logger = logging.getLogger(__name__)


class TemporaryFileServerConnector:

    @dataclass
    class TemporaryFileServerConnectorConfig:
        server_name: str

    @dataclass
    class Config(HttpServerConnector.Config, TemporaryFileServerConnectorConfig):
        pass

    Context = HttpServerConnector.Context

    def __init__(self, config: Config, context: Context):
        self._config = config
        self._context = context
        self._connector = HttpServerConnector(config, context)

    async def get_file(self, file_id: str, range_: Optional[int] = None) -> bytes:
        args = {"id": file_id}
        headers = self._get_auth_header()
        if range_ is not None:
            args["Range"] = range_

        return await self._connector.get("/file/get", args, headers=headers)

    async def get_file_info(self, file_id: str) -> dict:
        args = {"id": file_id}
        answer = await self._connector.get("/file/info", args, headers=self._get_auth_header())
        return answer["result"]

    async def add_file(self, file: bytes, filename: str, is_create_thumbnail: bool = False) -> dict:
        args = {"filename": filename, "is_create_thumbnail": 'true' if is_create_thumbnail else 'false'}
        answer = await self._connector.post("/file/add", payload=file, url_query=args, headers=self._get_auth_header())
        return answer["result"]

    async def delete_file(self, file_id: str) -> dict:
        args = {"id": file_id}
        answer = await self._connector.post("/file/delete", payload=None,
                                            url_query=args, headers=self._get_auth_header())
        return answer["result"]

    def _get_auth_header(self) -> dict[str, str]:
        return {"server_name": self._config.server_name}
