#  Copyright (C) 2021-2025
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors:
#       - Vasya Svintsov <v.svintsov@techokert.ru>,
#       - Alexander Medvedev <a.medvedev@abm-jsc.ru>,
#       - Albakov Ruslan <r.albakov@abm-jsc.ru>,
import traceback
from functools import wraps

from botocore.exceptions import ConnectionError
from dict_caster import ItemException
from file_storage.exceptions import PathDoesNotExists, FileStorageFull
from http_tools import ErrorAnswer, Answer, HttpStatusCode
from http_tools.http_server import Handler
from sqlalchemy.exc import SQLAlchemyError


def error_middleware(handler: Handler):
    @wraps(handler)
    async def wrapper(*args, **kwargs) -> Answer:
        try:
            return await handler(*args, **kwargs)
        except (LookupError, FileNotFoundError, PathDoesNotExists) as er:
            traceback.print_exc()
            return ErrorAnswer(er, status=HttpStatusCode.NotFound)
        except (ItemException, KeyError, ValueError) as er:
            traceback.print_exc()
            return ErrorAnswer(repr(er), status=HttpStatusCode.UnprocessableEntity)
        except (ConnectionError, SQLAlchemyError) as er:
            traceback.print_exc()
            return ErrorAnswer(repr(er), status=HttpStatusCode.BadGateway)
        except FileStorageFull as er:
            traceback.print_exc()
            return ErrorAnswer(repr(er), status=HttpStatusCode.InsufficientStorage)
    return wrapper
