#  Copyright (C) 2024
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Vasya Svintsov <v.svintsov@techokert.ru>

from dataclasses import dataclass

from http_tools import AbmServiceConnector
from init_helpers.dict_to_dataclass import get_converter


class NetrisProviderConnector:
    Config = AbmServiceConnector.Config

    @dataclass
    class Context(AbmServiceConnector.Context):
        project_name: str

    def __init__(self, config: Config, context: Context) -> None:
        self.config = config
        self.context = context
        self._connector = AbmServiceConnector(config, context)

    async def get_camera_outer_id_to_inner_id(self, outer_ids: list[str]) -> dict[str, int]:
        camera_outer_id_to_inner_id = await self._connector.post(
            '/camera_matching/outer_id_to_inner_id/get', {'ids': outer_ids}, headers=self._prepare_headers(),
        )
        return get_converter(dict[str, int])(camera_outer_id_to_inner_id)

    async def get_camera_inner_id_to_outer_id(self, inner_ids: list[int]) -> dict[int, str]:
        camera_inner_id_to_outer_id = await self._connector.post(
            '/camera_matching/inner_id_to_outer_id/get', {'ids': inner_ids}, headers=self._prepare_headers(),
        )
        return get_converter(dict[int, str])(camera_inner_id_to_outer_id)

    async def get_snapshot_for_outer_camera(self, id: str) -> bytes:
        snapshot = await self._connector.get(
            '/snapshot/outer_camera/get', {"id": id}, headers=self._prepare_headers()
        )

        assert isinstance(snapshot, bytes)

        return snapshot

    async def get_snapshot_for_inner_camera(self, id: int) -> bytes:
        snapshot = await self._connector.get(
            '/snapshot/inner_camera/get', {"id": id}, headers=self._prepare_headers()
        )

        assert isinstance(snapshot, bytes)

        return snapshot

    async def generate_archive(self, id: int, start_at: int, end_at: int) -> str:
        archive_id = await self._connector.post(
            '/archive/generate',{'id': id, 'start_at': start_at, 'end_at': end_at}, headers=self._prepare_headers(),
        )
        return archive_id

    async def get_archive_url(self, archive_id: str) -> str:
        archive_url = await self._connector.get(
            '/archive/get', {'archive_id': archive_id}, headers=self._prepare_headers(),
        )
        return archive_url

    def _prepare_headers(self) -> dict[str, str]:
        return {"server_name": self.context.project_name}
