"""HTTP connector for MentorTools Deep Agent API."""

from dataclasses import dataclass
from typing import Any

import aiohttp
from deep_agent_connector.exceptions import DeepAgentAPIError
from http_tools import AbmServiceConnector
from pydantic import BaseModel, Field


class WrappedResponse(BaseModel):
    """API response wrapper (mirrors server-side response_wrapper.py)."""

    done: bool
    result: Any | None = None
    error: Any | None = None
    error_type: str | None = None


class HealthResponse(BaseModel):
    status: str
    service: str
    workers: dict[str, Any] | None = Field(None, description="Background workers status")


class RunCreateRequest(BaseModel):
    user: dict[str, int] = Field(..., description="User context with id and portal_id")
    assistant: dict[str, int] = Field(..., description="Assistant context with id")
    chat: dict[str, int] = Field(..., description="Chat context with id, chat_session_id, bot_chat_member_id")
    message: str = Field(..., description="User message to send to the agent")
    request_id: str | None = Field(None, description="Optional request ID for tracing")


class RunCreateResponse(BaseModel):
    run_id: str = Field(..., description="Unique ID of the created run")
    status: int = Field(201, description="HTTP status code")


@dataclass
class DeepAgentConnector:
    """Async HTTP client for the Deep Agent service."""

    base_url: str
    api_key: str | None = None
    timeout_s: float = 30.0

    def __post_init__(self) -> None:
        self._session: aiohttp.ClientSession | None = None
        self._connector: AbmServiceConnector | None = None

    async def __aenter__(self) -> "DeepAgentConnector":
        headers: dict[str, str] = {}
        if self.api_key:
            headers["X-API-Key"] = self.api_key

        self._session = aiohttp.ClientSession(headers=headers)
        self._connector = AbmServiceConnector(
            AbmServiceConnector.Config(url=self.base_url.rstrip("/"), timeout_sec=self.timeout_s),
            AbmServiceConnector.Context(session=self._session),
        )
        return self

    async def __aexit__(self, exc_type, exc, tb) -> None:  # noqa: ANN001
        self._connector = None
        if self._session:
            await self._session.close()
            self._session = None

    def _get_connector(self) -> AbmServiceConnector:
        if not self._connector:
            raise RuntimeError("Connector is not initialized. Use `async with DeepAgentConnector(...)`.")
        return self._connector

    async def health(self, *, request_id: str | None = None) -> HealthResponse:
        """GET /health"""
        connector = self._get_connector()
        headers = {"X-Request-ID": request_id} if request_id else None

        result = await connector.get("/health", headers=headers)
        # Server typically wraps responses, but allow raw responses too.
        if isinstance(result, dict) and "done" in result:
            wrapper = WrappedResponse.model_validate(result)
            if not wrapper.done or wrapper.result is None:
                raise DeepAgentAPIError(
                    error_type=wrapper.error_type or "UnknownError",
                    message=str(wrapper.error or "Health request failed"),
                )
            return HealthResponse.model_validate(wrapper.result)

        return HealthResponse.model_validate(result)

    async def create_run(
        self,
        *,
        user_id: int,
        portal_id: int,
        assistant_id: int,
        message: str,
        chat_id: int,
        chat_session_id: int,
        bot_chat_member_id: int,
        request_id: str | None = None,
    ) -> RunCreateResponse:
        """POST /runs"""
        connector = self._get_connector()
        payload = RunCreateRequest(
            user={"id": user_id, "portal_id": portal_id},
            assistant={"id": assistant_id},
            chat={
                "id": chat_id,
                "chat_session_id": chat_session_id,
                "bot_chat_member_id": bot_chat_member_id,
            },
            message=message,
            request_id=request_id,
        )
        headers = {"X-Request-ID": request_id} if request_id else None
        result = await connector.post("/runs", payload=payload.model_dump(), headers=headers)
        if isinstance(result, dict) and "done" in result:
            wrapper = WrappedResponse.model_validate(result)
            if not wrapper.done or wrapper.result is None:
                raise DeepAgentAPIError(
                    error_type=wrapper.error_type or "UnknownError",
                    message=str(wrapper.error or "Create run failed"),
                )
            return RunCreateResponse.model_validate(wrapper.result)

        return RunCreateResponse.model_validate(result)
