#  Copyright (C) 2025
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Mike Orlov <m.orlov@abm-jsc.ru>
import functools
from dataclasses import dataclass
from typing import Mapping

import yarl
from frozendict import frozendict
from init_helpers import Jsonable, AnyType

from .spec_ref import SpecRef
from .spec_resource import SpecResource


@dataclass(frozen=True, slots=True)
class Info(SpecResource):
    """Info https://github.com/OAI/OpenAPI-Specification/blob/main/versions/3.1.0.md#info-object"""
    title: str
    version: str
    summary: str = ''
    description: str = ''
    terms_of_service: yarl.URL | None = None
    # contact: Contact  # TODO: implement
    # license: License  # TODO: implement

    def get_spec_dependencies(self) -> frozenset[SpecResource]:
        return frozenset()

    def get_spec_dict(self, dependency_to_ref: Mapping[SpecResource | AnyType, SpecRef]) -> frozendict[str, Jsonable]:
        result = {'title': self.title, 'version': self.version}
        result |= {"summary": self.summary} if self.summary else {}
        result |= {"description": self.description} if self.description else {}
        result |= {"terms_of_service": str(self.terms_of_service)} if self.terms_of_service else {}
        return frozendict(result)

    @functools.cache
    def _get_repr_parts(self) -> tuple[str, ...]:
        parts = [f'title={self.title!r}', f'version={self.version!r}']
        parts += [f'summary={self.summary!r}'] if self.summary else []
        parts += [f'description={self.description!r}'] if self.description else []
        parts += [f'terms_of_service={str(self.terms_of_service)!r}'] if self.terms_of_service else []
        return tuple(parts)

    def __repr__(self):
        return f'{self.__class__.__name__}({", ".join(self._get_repr_parts())})'

    __str__ = __repr__
