from dataclasses import asdict, dataclass
from logging import Logger
from aiokafka import AIOKafkaProducer

from async_tools import AsyncOnStart, AsyncOnStop

from models.notification import Notification
from utils.abstract_encoder import AbstractEncoder


logger = Logger(__name__)


class NotificationConnector(AsyncOnStart, AsyncOnStop):
    @dataclass
    class Config:
        address: str
        max_request_size: int
        channel_name: str

    @dataclass
    class Context:
        encoder: AbstractEncoder

    _config: Config
    _producer: AIOKafkaProducer
    _context: Context

    def __init__(self, config: Config, context: Context) -> None:
        self._config = config
        self._context = context

    async def _on_start(self) -> None:
        self._producer = AIOKafkaProducer(
            bootstrap_servers=[self._config.address],
            max_request_size=self._config.max_request_size,
        )

    async def _on_stop(self) -> None:
        await self._producer.stop()

    async def send(self, notification: Notification) -> bool:
        logger.debug(f'write to kafka channel "{self._config.channel_name}": {notification}')
        payload = self._context.encoder.encode(asdict(notification))
        result = await self._producer.send(self._config.channel_name, value=payload)
        logger.debug(f'write to kafka channel "{self._config.channel_name}" done, result {result}')
        return True
