#  Copyright (C) 2021
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Vasya Svintsov <v.svintsov@techokert.ru>, Alexander Medvedev <a.medvedev@abm-jsc.ru>

from sqlalchemy import Integer, String, BigInteger, text
from sqlalchemy_tools.entity_helpers.decorator import sqlalchemy_dataclass
from sqlalchemy_tools.entity_helpers.fields import sql_field, sql_relation_field, get_column_from_field
from sqlalchemy_tools.entity_helpers.fk_keys import CascadeForeignKey
from sqlalchemy_tools.entity_helpers.sqlalchemy_base import sqlalchemy_mapper_registry

from ..tools.get_timestamp import get_current_timestamp
from ..tools.hashes import generate_md5_hash, generate_sha512_hash


@sqlalchemy_dataclass(sqlalchemy_mapper_registry)
class FileMetadata:
    storage_file_name: str = sql_field(String, primary_key=True, nullable=False, allow_init_primary_key=True)
    relative_storage_path: str = sql_field(String, nullable=False)

    file_md5: str = sql_field(String, nullable=False)
    file_sha512: str = sql_field(String, nullable=False)

    file_size: int = sql_field(Integer, nullable=False)
    full_file_name: str = sql_field(String, nullable=False)
    file_extension: str = sql_field(String, nullable=True)

    uploaded_at: int = sql_field(BigInteger,
                                 nullable=False,
                                 default=get_current_timestamp,
                                 server_default=text('ROUND(EXTRACT(EPOCH FROM NOW())*1000)'),
                                 )

    parent_id: str = sql_field(String, CascadeForeignKey(get_column_from_field(storage_file_name)), nullable=True)
    thumbnail: 'FileMetadata' = sql_relation_field('FileMetadata', use_list=False, lazy='noload')

    @classmethod
    def prepare(cls, file_content: bytes, file_name: str, file_extension: str | None = None) -> 'FileMetadata':
        file_size = len(file_content)
        file_md5 = generate_md5_hash(file_content)
        file_sha512 = generate_sha512_hash(file_content)
        storage_path = file_sha512[:3]
        storage_file_name = f'{file_md5}_{file_size}'

        file_metadata = FileMetadata(storage_file_name=storage_file_name,
                                     relative_storage_path=storage_path,
                                     file_md5=file_md5,
                                     file_sha512=file_sha512,
                                     full_file_name=file_name,
                                     file_extension=file_extension,
                                     file_size=file_size,
                                     )
        file_metadata.thumbnail = None
        return file_metadata
