#  Copyright (C) 2024
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Vasya Svintsov <v.svintsov@techokert.ru>
from dataclasses import dataclass

import aiohttp.hdrs
from aiohttp import ClientSession, ClientTimeout

from .tools.extension_to_mime_type import mime_type_to_extension
from .tools.get_filename_from_headers import get_filename_from_headers


@dataclass
class FileMetadata:
    filename: str | None
    extension: str | None
    size: int | None
    etag: str | None

    @property
    def clear_etag(self) -> str | None:
        if self.etag:
            return self.etag.replace('"', '')


class FileDownloader:
    @dataclass
    class Config:
        timeout_sec: float = 100

    @dataclass
    class Context:
        session: ClientSession

    def __init__(self, config: Config, context: Context) -> None:
        self.context = context
        self.config = config

    async def head(self, file_url: str) -> FileMetadata:
        async with self.context.session.head(file_url, timeout=ClientTimeout(self.config.timeout_sec)) as answer:
            answer.raise_for_status()
            return FileMetadata(get_filename_from_headers(answer.headers),
                                mime_type_to_extension.get(answer.headers.get(aiohttp.hdrs.CONTENT_TYPE)),
                                answer.content_length,
                                answer.headers.get(aiohttp.hdrs.ETAG),
                                )

    async def get(self, file_url: str) -> tuple[str | None, bytes]:
        async with self.context.session.get(file_url, timeout=ClientTimeout(self.config.timeout_sec)) as answer:
            answer.raise_for_status()
            return get_filename_from_headers(answer.headers), await answer.read()
