#  Copyright (C) 2021
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Vasya Svintsov <v.svintsov@techokert.ru>

import logging
import os
from concurrent.futures import ThreadPoolExecutor
from dataclasses import dataclass
from pathlib import Path
from typing import Optional

import aiofiles
from aiofiles.os import remove
from aiofiles.ospath import exists

from ..abstract_file_storage import AbstractFileStorage, AbstractStorageStatus
from ..exceptions import FileAlreadyExists, PathDoesNotExists, InsecurePath

logger = logging.getLogger(__name__)


@dataclass
class DiscStorageStatus(AbstractStorageStatus):
    pass


class DiscFileStorage(AbstractFileStorage):
    @dataclass(kw_only=True)
    class Config(AbstractFileStorage.Config):
        location: str
        max_workers: Optional[int] = None

    Context = AbstractFileStorage.Context

    def __init__(self, config: Config, context: Context) -> None:
        super().__init__(config, context)
        self._thread_pool_executor = ThreadPoolExecutor(max_workers=self.config.max_workers)
        self._set_storage_status()
        logger.info(f"{type(self).__name__} inited. Location: {self.config.location}")

    async def _save(self, key: str, value: bytes, allow_rewrite: bool = False) -> None:
        file_path = self._get_file_path(key)

        dir_path = file_path.parent
        if not await aiofiles.ospath.exists(dir_path, executor=self._thread_pool_executor):
            logger.info(f"Path = {dir_path} does not exists. Will be created")
            await aiofiles.os.makedirs(dir_path)

        if await aiofiles.ospath.exists(file_path, executor=self._thread_pool_executor) and not allow_rewrite:
            raise FileAlreadyExists(file_path)

        logger.info(f"Saving file along the path: {file_path}")
        async with aiofiles.open(file_path, "wb", executor=self._thread_pool_executor) as file:
            await file.write(value)

    async def _load(self, key: str, offset: int = 0, size: int = -1) -> bytes:
        file_path = self._get_file_path(key)

        if not await aiofiles.ospath.exists(file_path, executor=self._thread_pool_executor):
            raise PathDoesNotExists(file_path)

        if not os.path.realpath(file_path).startswith(os.path.realpath(file_path.parent)):
            raise InsecurePath(file_path)

        logger.info(f"Loading file along the path: {file_path}")
        async with aiofiles.open(file_path, "rb", executor=self._thread_pool_executor) as file:
            await file.seek(offset)
            content = await file.read(size)
        return content

    async def _delete(self, key: str) -> None:
        file_path = self._get_file_path(key)

        if not await aiofiles.ospath.exists(file_path, executor=self._thread_pool_executor):
            raise PathDoesNotExists(file_path)

        logger.info(f"Deleting file along the path: {file_path}")
        await aiofiles.os.remove(file_path, executor=self._thread_pool_executor)

        dir_path = file_path.parent
        if not await aiofiles.os.listdir(dir_path, executor=self._thread_pool_executor):
            await aiofiles.os.rmdir(dir_path, executor=self._thread_pool_executor)

    async def _check_file_existence(self, key: str) -> bool:
        file_path = self._get_file_path(key)
        return await aiofiles.ospath.exists(file_path, executor=self._thread_pool_executor)

    def _get_file_path(self, key: str) -> Path:
        return Path(self.config.location, self.config.root_dir, *self._split_key(key))

    def _get_storage_status(self) -> DiscStorageStatus:
        used = 0
        for file_path in Path(self.config.location,  self.config.root_dir).rglob('*'):
            try:
                used += file_path.stat().st_size if file_path.is_file() else os.path.getsize(file_path)
            except (PermissionError, OSError):
                continue
        free = max(self.config.max_capacity_b - used, 0.0)
        return DiscStorageStatus(total_b=self.config.max_capacity_b,
                                 used_b=used,
                                 free_b=free)

    def _set_storage_status(self) -> None:
        self.status = self._get_storage_status()

    def _get_file_size(self, key: str) -> int:
        file_path = self._get_file_path(key)
        return file_path.stat().st_size
