#  Copyright (C) 2022
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Vasya Svintsov <v.svintsov@techokert.ru>, Alexandr Medvedev <a.medvedev@abm-jsc.ru>

import logging
import time
from asyncio import Lock
from dataclasses import dataclass
from typing import Optional, Dict, List, Any

from async_tools import AsyncOnStart
from dict_caster import DictCaster, Item
from furl import furl

from .defs import TicketListResponse, TicketType
from .http_tech_support_connector import HttpTechSupportConnector
from .tools.cast_jsonable_to_dataclass import cast_jsonable_to_dataclass


logger = logging.getLogger(__name__)


class TechSupportConnector(AsyncOnStart):
    @dataclass
    class AuthConfig:
        username: str
        password: str

    @dataclass
    class Config(HttpTechSupportConnector.Config, AuthConfig):
        ...

    Context = HttpTechSupportConnector.Context

    @dataclass(frozen=True)
    class Access:
        token: str

        @property
        def auth_headers(self) -> Dict[str, str]:
            return {"Authorization": f"Bearer {self.token}"}

    def __init__(self, config: Config, context: Context):
        self._config = config

        self._access_token: Optional[str] = None
        self._refresh_token: Optional[str] = None
        lifetime_shift_in_sec = 5 * 60
        self._access_token_lifetime_in_sec = 15 * 60 - lifetime_shift_in_sec
        self._refresh_token_lifetime_in_sec = 30 * 24 * 60 * 60 - lifetime_shift_in_sec
        self._last_login_at = .0
        self._last_refresh_token_at = .0

        self._access_lock = Lock()
        self._http_connector = HttpTechSupportConnector(config, context)

    async def _login(self) -> None:
        response = await self._http_connector.post_json(
            '/login', payload={"username": self._config.username, "password": self._config.password})
        self._access_token, self._refresh_token = DictCaster(
            Item("access_token", str),
            Item("refresh_token", str)
        ).cast_and_return(response)
        logger.info("Login successful")

    async def _refresh_access_token(self) -> None:
        response = await self._http_connector.get("/login", headers=self.Access(self._refresh_token).auth_headers)
        self._access_token = DictCaster(Item("access_token", str)).cast_and_return(response)
        logger.info("Refresh access_token successful")

    async def _login_and_commit_time(self) -> None:
        await self._login()
        current_at = time.time()
        self._last_login_at = current_at
        self._last_refresh_token_at = current_at
        logger.info(f"last_login_at: {self._last_login_at}")

    async def _refresh_access_token_and_commit_time(self) -> None:
        await self._refresh_access_token()
        self._last_refresh_token_at = time.time()
        logger.info(f"last_refresh_token_at: {self._last_refresh_token_at}")

    async def _get_access(self) -> Access:
        async with self._access_lock:
            current_at = time.time()
            if current_at - self._last_login_at > self._refresh_token_lifetime_in_sec:
                await self._login_and_commit_time()
            elif current_at - self._last_refresh_token_at > self._access_token_lifetime_in_sec:
                await self._refresh_access_token_and_commit_time()
        return self.Access(self._access_token)

    async def _on_start(self) -> None:
        await self._get_access()
        logger.info(f"{type(self).__name__} started")

    async def get_ticket_list(self,
                              fields: Optional[List[str]] = None,
                              filters: Optional[Dict[str, Any]] = None,
                              limit: int = 50,
                              since: Optional[str] = None
                              ) -> TicketListResponse:
        payload = {"limit": limit}
        if fields is not None:
            payload["fields"] = fields
        if filters is not None:
            payload["filters"] = filters
        if since is not None:
            payload["since"] = since

        response = await self._http_connector.get_json(
            "/tickets", payload=payload, headers=(await self._get_access()).auth_headers)

        return cast_jsonable_to_dataclass(response, cls=TicketListResponse)

    async def get_ticket(self,
                         uuid: str,
                         type_: TicketType
                         ) -> Dict[str, Any]:
        path = furl("/tickets") / type_ / uuid
        response = await self._http_connector.get(path.url, headers=(await self._get_access()).auth_headers)
        return response

    async def update_ticket(self,
                            uuid: str,
                            type_: TicketType,
                            values_to_update: Dict[str, Any]
                            ) -> Dict[str, Any]:
        path = furl("/tickets") / type_ / uuid
        response = await self._http_connector.patch_json(
            path.url, payload=values_to_update, headers=(await self._get_access()).auth_headers)
        return response

    async def create_ticket(self,
                            values_to_create: Dict[str, Any]
                            ) -> Dict[str, Any]:
        response = await self._http_connector.post_json(
            "/tickets", payload=values_to_create, headers=(await self._get_access()).auth_headers)
        return response

    async def get_ticket_stages(self,
                                uuid: str,
                                type_: TicketType
                                ) -> Dict[str, Any]:
        path = furl("/tickets") / type_ / uuid / "stages"
        response = await self._http_connector.get(path.url, headers=(await self._get_access()).auth_headers)
        return response
