from dataclasses import dataclass

from jinja2 import Template, Environment

from extended_logger import get_logger
from template.email_renderer import EmailRenderer, Subject, Body
from translation_loader import TranslationLoader

logger = get_logger(__file__)


@dataclass(frozen=True)
class EmailTemplate:
    subject: Template
    body: Template

    def render(self, **template_variables) -> tuple[Subject, Body]:
        subject = self.subject.render(**template_variables)
        body = self.body.render(**template_variables)
        return subject, body


class WeblateEmailRenderer(EmailRenderer):
    @dataclass
    class Context:
        project_name: str
        translation_loader: TranslationLoader

    @dataclass
    class Config:
        weblate_component_name: str = 'email_renderer'
        weblate_translation_language: str = 'ru'

    def __init__(self, config: Config, context: Context):
        self.config = config
        self.context = context
        self._environment = Environment()
        self._email_type_name_to_templates: dict[str, EmailTemplate] = {}

    async def load_templates_for_emails(self, email_type_names: list[str]):
        logger.debug('get_translations_file_content')
        email_type_name_to_pattern = await self.context.translation_loader.get_translations_file_content(
            project=self.context.project_name.lower(), component=self.config.weblate_component_name,
            language=self.config.weblate_translation_language
        )
        for email_type_name in email_type_names:
            pattern = email_type_name_to_pattern[email_type_name]
            subject_template = self._environment.from_string(pattern["subject"])
            body_template = self._environment.from_string(pattern["body"])
            email_template = EmailTemplate(subject_template, body_template)
            self._email_type_name_to_templates[email_type_name] = email_template

    def render_email(self, email_type_name: str, **kwargs) -> tuple[Subject, Body]:
        return self._email_type_name_to_templates[email_type_name].render(**kwargs)
