"""
template - jinja2.Template
pattern  - entity_tools string for template in jinja2 format
"""
import html
import datetime
from dataclasses import dataclass, field
from typing import Type, Union, Optional, Any, Iterable

from init_helpers.dict_to_dataclass import NoValue
from jinja2 import Template, Environment

from extended_logger import get_logger
from time_tools import TimestampSec

from entity_view.viewable_entity import ViewableEntity
from entity_field import NoDefault

logger = get_logger(__name__)
SPACE = " "
# SPACE = "&nbsp;"


@dataclass
class EntityRenderPattern:
    title: str
    key_to_title: dict[str, str] = field(default_factory=dict)
    key_to_preprocessor: dict[str, str] = field(default_factory=dict)
    relation_to_title: dict[str, str] = field(default_factory=dict)

    create_description: str = (
        "Создан {{ render_title(target) }}:"
        "{{ padded(render_attributes(target)) }}"
        "{{ padded(render_relations(target)) }}"
    )
    update_description: str = (
        "Изменён {{ render_title(target) }}:"
        "{{ padded(render_attribute_updates(target)) }}"
        "{{ padded(render_relation_deltas(target)) }}"
    )
    delete_description: str = "Удалён {{ render_title(target) }}"

    attribute_updates: str = (
        "{% for key, title in key_to_title.items() %}"
        "{% if title %}"
        "{{ render_attribute_update(target, key) }}"
        "{% endif %}"
        "{% endfor %}"
    )
    attribute_update: str = '\n* {{ key }}: с "{{ old_value }}" на "{{ new_value }}"'

    relation_deltas: str = (
        "{% for relation in relation_to_title %}"
        "{{ render_relation_delta(target, relation) }}"
        "{% endfor %}"
    )
    relation_delta: str = (
        "{% set max_amount = 5 %}"
        "{% if creates or updates or deletes %}\n@ {{ key }}"
        "{% if len(creates) > max_amount %}, создано {{ len(creates) }} (показано {{ max_amount }}){% endif %}" 
        "{% if len(updates) > max_amount %}, изменено {{ len(updates) }} (показано {{ max_amount }}) {% endif %}" 
        "{% if len(deletes) > max_amount %}, удалено {{ len(deletes) }} (показано {{ max_amount }}) {% endif %}"
        ":"
        + (
            "{% for create in creates[:max_amount] %}{{ padded('\n+ ' + render_entity(create)) }} {% endfor %}"
            "{% if len(creates) > max_amount %}\n ... {% endif %}"
            "{% for update in updates[:max_amount] %}{{ padded('\n~ ' + render_entity(update)) }} {% endfor %}"
            "{% if len(updates) > max_amount %}\n ... {% endif %}"
            "{% for delete in deletes[:max_amount] %}{{ padded('\n- ' + render_entity(delete)) }} {% endfor %}"
            "{% if len(delete) > max_amount %}\n ... {% endif %}"
        ) +
        "{% endif %}"
    )

    attributes: str = (
        "{% for key, title in key_to_title.items() %}"
        "{% if title %}"
        "{{ render_attribute(target, key) }}"
        "{% endif %}"
        "{% endfor %}"
    )
    attribute: str = '{% if value %}\n* {{ key }}: "{{ value }}"{% endif %}'
    relations: str = (
        "{% for relation in relation_to_title %}"
        "{{ render_relation(target, relation) }}"
        "{% endfor %}"
    )
    relation: str = (
        '{% if values %}' + (
            '\n@ {{ key }}: {% for val in values %}{{ padded("\n+ " + render_entity(val)) }} {% endfor %}'
        ) +
        "{% endif %}"
    )


@dataclass
class _EntityRenderTemplate:
    title: Template
    key_to_title: dict[str, str]
    key_to_preprocessor_template: dict[str, Template]
    relation_to_title: dict[str, str]
    create_description: Template
    update_description: Template
    delete_description: Template
    attribute_updates: Template
    attribute_update: Template
    relation_deltas: Template
    relation_delta: Template
    attributes: Template
    attribute: Template
    relations: Template
    relation: Template

    @classmethod
    def from_pattern(cls, pattern: EntityRenderPattern, environment: Environment):
        return cls(
            title=environment.from_string(pattern.title),
            create_description=environment.from_string(pattern.create_description),
            update_description=environment.from_string(pattern.update_description),
            delete_description=environment.from_string(pattern.delete_description),
            key_to_title=pattern.key_to_title,
            key_to_preprocessor_template={
                key: environment.from_string(preprocessor)
                for key, preprocessor in pattern.key_to_preprocessor.items()
            },
            relation_to_title=pattern.relation_to_title,
            attribute_updates=environment.from_string(pattern.attribute_updates),
            attribute_update=environment.from_string(pattern.attribute_update),
            relation_deltas=environment.from_string(pattern.relation_deltas),
            relation_delta=environment.from_string(pattern.relation_delta),
            attributes=environment.from_string(pattern.attributes),
            attribute=environment.from_string(pattern.attribute),
            relations=environment.from_string(pattern.relations),
            relation=environment.from_string(pattern.relation),
        )


class Wrapper:
    def __init__(
            self, entity_type_to_template: dict[Type[ViewableEntity], _EntityRenderTemplate], target: ViewableEntity,
            preloaded_entity_type_to_primary_keys_to_entity: Optional[dict[Type[ViewableEntity], dict[tuple, ViewableEntity]]] = None
    ) -> None:
        preloaded_entity_type_to_primary_keys_to_entity = preloaded_entity_type_to_primary_keys_to_entity or {}
        self.entity_type_to_template = entity_type_to_template
        self.entity_type_to_primary_keys_to_entity = {
            **preloaded_entity_type_to_primary_keys_to_entity,
            **target.get_type_to_primary_keys_to_entity()
        }

    @staticmethod
    def _primary_keys_to_tuple(**primary_keys: Any) -> tuple:
        sorted_key_value_pairs = sorted(primary_keys.items())
        sorted_key_values = tuple(value for key, value in sorted_key_value_pairs)
        return sorted_key_values

    def _resolve(self, type_: Type[ViewableEntity], **primary_keys: Any) -> ViewableEntity:
        try:
            primary_keys_to_entity = self.entity_type_to_primary_keys_to_entity[type_]
            primary_keys_tuple = tuple(primary_keys.get(name) for name in type_.get_primary_key_names())
            entity = primary_keys_to_entity[primary_keys_tuple]
            return entity
        except KeyError as e:
            logger.error(f"_resolve failed to found %s with %s: %s", (type_, primary_keys, repr(e)))
            raise

    def _get_template_by_entity(self, target: Union[ViewableEntity, Type[ViewableEntity]]) -> _EntityRenderTemplate:
        entity_type = target.get_entity_type()
        return self.entity_type_to_template[entity_type]

    def _render(self, template: Template, **kwargs) -> str:
        kwargs["get"] = self._get
        kwargs["resolve"] = self._resolve
        kwargs["render_title"] = self._render_title
        kwargs["render_attribute_updates"] = self._render_attribute_updates
        kwargs["render_attribute_update"] = self._render_attribute_update
        kwargs["render_relation_deltas"] = self._render_relation_deltas
        kwargs["render_relation_delta"] = self._render_relation_delta
        kwargs["render_attributes"] = self._render_attributes
        kwargs["render_attribute"] = self._render_attribute
        kwargs["render_relations"] = self._render_relations
        kwargs["render_relation"] = self._render_relation
        kwargs["render_entity"] = self._render_entity
        for entity_type in self.entity_type_to_template:
            kwargs[entity_type.__name__] = entity_type
        return template.render(**kwargs)

    # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
    # -> utils

    def _get(self, target: ViewableEntity, key: str):
        val = getattr(target, key, NoValue)
        if val in (NoValue, NoDefault):
            parent_view = target.get_parent_view()
            if parent_view not in (NoValue, NoDefault):
                val = getattr(parent_view, key, NoValue)

        try:
            val = val.old
        except AttributeError:
            pass
        # logger.trace(f'_get: %s, %s => %s', type(target).__name__, key, val)
        if val in (NoValue, NoDefault):
            val = None
        else:
            template = self._get_template_by_entity(target)
            if (preprocessor_template := template.key_to_preprocessor_template.get(key)) is not None:
                val = self._render(preprocessor_template, val=val)
        return val

    # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
    # -> render

    def _render_title(self, target: ViewableEntity) -> str:
        # logger.trace(f'_render_title: %s', type(target).__name__, )
        template = self._get_template_by_entity(target)
        return self._render(template.title, target=target)  # TODO: add "get", think about missing params

    def _render_attribute_updates(self, target: ViewableEntity) -> str:
        # logger.trace(f'_render_attribute_updates: %s', type(target).__name__, )
        template = self._get_template_by_entity(target)
        return self._render(template.attribute_updates, target=target, key_to_title=template.key_to_title)

    def _render_attribute_update(self, target: ViewableEntity, key) -> str:
        # logger.trace(f'_render_attribute_update: %s, %s', type(target).__name__, key)
        if key in target.get_primary_key_names():
            return ""
        val = getattr(target, key, NoValue)
        if val is NoValue:
            return ""
        template = self._get_template_by_entity(target)
        old_value = val.old
        new_value = val.new
        if (preprocessor_template := template.key_to_preprocessor_template.get(key)) is not None:
            old_value = self._render(preprocessor_template, val=old_value)
            new_value = self._render(preprocessor_template, val=new_value)
        title = template.key_to_title[key]
        return self._render(template.attribute_update, key=title, old_value=old_value, new_value=new_value)
        # escape

    def _render_relation_deltas(self, target: ViewableEntity) -> str:
        # logger.trace(f'_render_relation_deltas: %s', type(target).__name__, )
        template = self._get_template_by_entity(target)
        res = self._render(template.relation_deltas, target=target, relation_to_title=template.relation_to_title)
        return res

    def _render_relation_delta(self, target: ViewableEntity, key: str) -> str:
        # logger.trace(f'_render_relation_delta: %s, %s', type(target).__name__, key)
        val = getattr(target, key, NoValue)

        if val is NoValue:
            return ""

        template = self._get_template_by_entity(target)
        res = self._render(
            template.relation_delta,
            key=template.relation_to_title[key],
            creates=val.create, updates=val.update, deletes=val.delete
        )
        return res

    def _render_attributes(self, target: ViewableEntity) -> str:
        # logger.trace(f'_render_attributes: %s', type(target).__name__)
        template = self._get_template_by_entity(target)
        return self._render(template.attributes, target=target, key_to_title=template.key_to_title)

    def _render_attribute(self, target: ViewableEntity, key) -> str:
        # logger.trace(f'_render_attribute: %s, %s', type(target).__name__, key)
        val = getattr(target, key, NoValue)
        if val is NoValue:
            return ""
        template = self._get_template_by_entity(target)
        if (preprocessor_template := template.key_to_preprocessor_template.get(key)) is not None:
            val = self._render(preprocessor_template, val=val)
        title = template.key_to_title[key]
        return self._render(template.attribute, key=title, value=val)
        # escape

    def _render_relations(self, target: ViewableEntity) -> str:
        # logger.trace(f'_render_relations: %s', type(target).__name__)
        template = self._get_template_by_entity(target)
        return self._render(template.relations, target=target, relation_to_title=template.relation_to_title)

    def _render_relation(self, target: ViewableEntity, key: str):
        # logger.trace(f'_render_relation: %s, %s', type(target).__name__, key)
        val = getattr(target, key, NoValue)

        if val is NoValue:
            return ""

        template = self._get_template_by_entity(target)
        result = self._render(template.relation, key=template.relation_to_title[key], values=val)
        return result

    def _render_entity(self, target: ViewableEntity):
        # logger.trace(f'_render_entity: %s', type(target).__name__)
        template = self._get_template_by_entity(target)

        if target.is_insert_view():
            description = template.create_description
        elif target.is_delta_view():
            description = template.update_description
        elif target.is_delete_view():
            description = template.delete_description
        else:
            raise TypeError(f"Unexpected type: {type(target)}")

        return self._render(description, target=target)

    def render_entity(self, target: ViewableEntity):
        return self._render_entity(target)


class DeltaViewRenderer:
    def __init__(self) -> None:
        self.preloaded_entity_type_to_primary_keys_to_entity: dict[Type[ViewableEntity], dict[tuple, ViewableEntity]] = {}
        logger.debug(f"init DeltaViewRenderer")
        self.entity_type_to_template: dict[Type[ViewableEntity], _EntityRenderTemplate] = {}
        self.entity_type_name_to_type: dict[str, Type[ViewableEntity]] = {}
        # self._entity = _DictWrapper(self.entity_type_name_to_type)
        self._environment = self._init_environment()

    def add_preloaded_entity(self, entity: ViewableEntity) -> None:
        self.preloaded_entity_type_to_primary_keys_to_entity.update(entity.get_type_to_primary_keys_to_entity())

    def add_preloaded_entities(self, entities: Iterable[ViewableEntity]) -> None:
        for entity in entities:
            self.add_preloaded_entity(entity)

    def _init_environment(self):
        environment = Environment()
        environment.globals["len"] = len
        environment.globals["date"] = self._date
        environment.globals["padded"] = self._padded
        return environment

    def register_pattern_for_entity_type(self, entity_type: Type[ViewableEntity], pattern: EntityRenderPattern) -> None:
        logger.debug(f"register_pattern_for_entity_type({entity_type.__name__}, {pattern})")
        self.entity_type_to_template[entity_type] = _EntityRenderTemplate.from_pattern(pattern, self._environment)
        self.entity_type_name_to_type[entity_type.__name__] = entity_type

    def render(self, delta_view: ViewableEntity) -> str:
        logger.debug(f"render delta_view: %s", delta_view)
        wrapper = Wrapper(
            self.entity_type_to_template, delta_view,
            self.preloaded_entity_type_to_primary_keys_to_entity
        )
        result = wrapper.render_entity(delta_view)
        result = f"<p>{self._escape_html(result)}</p>"
        logger.debug(f"render result: %s", result)
        return result

    @staticmethod
    def _escape_html(text: str) -> str:
        result = html.escape(text, quote=False)
        return result

    # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
    # -> utils

    @staticmethod
    def _padded(text: str, padding: str = SPACE):
        return text.replace("\n", "\n" + padding)

    @staticmethod
    def _date(timestamp: Optional[float]) -> Optional[str]:
        if timestamp is None:
            return

        return datetime.datetime.fromtimestamp(float(TimestampSec(timestamp)) / 1000).strftime('%Y-%m-%d')

# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #


# delta_view_chat_renderer = DeltaViewRenderer()
# delta_view_chat_renderer.register_pattern_for_entity_type(Contract, EntityRenderPattern(
#     title='Черновик контракта "{{ get(target,"id") }}"',
#     key_to_title={
#         "target_object_id": "Целевой объект",
#         "title": "Заголовок",
#         "region_id": "Регион",
#     },
#     relation_to_title={
#         "stages": "Этапы",
#         "responsible_persons": "Ответственные лица"
#     },
# ))
# delta_view_chat_renderer.register_pattern_for_entity_type(ContractStage, EntityRenderPattern(
#     title='Этап "{{ get(target,"title") }}"'
#           '(с {{ get(target,"start_at_epoch") }} по {{ get(target,"end_at_epoch") }})',
#     key_to_title={
#         "title": "Заголовок",
#         "price": "Этапы",
#         "calculation_version_id": "Версия расчёта",
#         "start_at_epoch": "Дата начала",
#         "end_at_epoch": "Дата конца"
#     },
# ))
# delta_view_chat_renderer.register_pattern_for_entity_type(ContractResponsiblePerson, EntityRenderPattern(
#     title='Ответственное лицо "{{ get(target,"name") }}"',  # TODO: resolve type_id
#     key_to_title={
#         "type_id": "Тип",
#         "name": "ФИО",
#         "phone_number": "Номер телефона",
#         "email": "Адрес электронной почты",
#         "address": "Адрес",
#         "additional_phone_number": "Дополнительный номер телефона",
#     },
# ))
#
#
# @dataclass
# class OptionalIntDelta:
#     old: Optional[int]
#     new: Optional[int]
#
#
# @dataclass
# class StrDelta:
#     old: str
#     new: str
#
#
# @dataclass
# class ContractStageDelta:
#     create: list[ContractStage[InsertView]]
#     update: list[ContractStage[DeltaView]]
#     delete: list[ContractStage[DeleteView]]
#
# # noinspection PyArgumentList
# delta = Contract[DeltaView](
#     id=67,
#     chat_id=OptionalIntDelta(old=396109, new=396118),
#     target_object_id=OptionalIntDelta(old=None, new=17),
#     title=StrDelta(old='Звёздный городок городской округ, Без оплаты, Контракт ТС, Тип 1', new='qwerty'),
#
#     stages=ContractStageDelta(
#         create=[
#             ContractStage[InsertView](
#                 id=-40191275, contract_id=26, price=82417693.7710104, calculation_version_id=-80437254,
#                 start_at_epoch=-58051633, end_at_epoch=85266909
#             ),
#             ContractStage[InsertView](
#                 id=123, title='ut est laboris aliqua et', contract_id=26, price=82417693.7710104,
#                 calculation_version_id=-80437254, start_at_epoch=-58051633,
#                 end_at_epoch=85266909, created_at=56347397
#             ),
#             ContractStage[InsertView](
#                 title='ut est laboris aliqua et', contract_id=26, price=82417693.7710104,
#                 calculation_version_id=-80437254, start_at_epoch=-58051633,
#                 end_at_epoch=85266909, created_at=56347397
#             )
#         ],
#         update=[],
#         delete=[
#             ContractStage[DeleteView](id=25)
#         ]
#     )
# )
# print(delta.stages.delete[0]._parent_view)
#
# res = delta_view_chat_renderer.render(delta)
# print(res)
