# Copyright (C) 2022
# ABM, Moscow
#
# UNPUBLISHED PROPRIETARY MATERIAL.
# ALL RIGHTS RESERVED.
#
# Authors: Alexander Medvedev <a.medvedev@abm-jsc.ru>

import logging
from dataclasses import dataclass
from typing import Optional

from aio_pika import RobustConnection, connect_robust
from aio_pika.abc import AbstractRobustChannel
from async_tools import AsyncOnStart, AsyncOnStop
from furl import furl

logger = logging.getLogger(__name__)


class RabbitMQConnector(AsyncOnStart, AsyncOnStop):
    @dataclass
    class Config:
        address: str
        username: str
        password: str

    def __init__(self, config: Config):
        self._config = config
        self._connection: Optional[RobustConnection] = None

    async def _on_start(self) -> None:
        if self._connection is not None:
            raise ConnectionError('Connection already exists')
        url = furl(scheme="amqp", netloc=self._config.address)
        url.set(username=self._config.username, password=self._config.password)
        self._connection = await connect_robust(url.tostr())
        logger.info('Connection successfully created')

    async def _on_stop(self) -> None:
        if self._connection is None:
            raise ConnectionError('Connection has been already terminated')
        await self._connection.close()
        self._connection = None
        logger.info('Connection successfully terminated')

    async def get_channel(self) -> AbstractRobustChannel:
        if self._connection is None:
            raise ConnectionError('There is no active connection')
        return await self._connection.channel()
